open Util

type action =
  Create of Dlg.dlg
| Append of string * (Dlg.state list)
| Extend_Bottom of string * (string list) * int * (Dlg.transition list)
| Extend_Top of string * (string list) * int * (Dlg.transition list)
| Replace of string * (Dlg.state list)
| Replace_Say of string * string * (Dlg.tlk_string)
| Add_Trans_Trigger of string * (string list)* string * (string list) 
| Add_Trans_Action 
    of string       (* filename *)
    * (string list) (* state labels *)
    * (int list)    (* transition #s, empty = all *)
    * string        (* new action text *) 
| Replace_State_Trigger of string * (string list)* string
| Add_State_Trigger of string * (string list)* string
| Set_Weight of string * string * int
| Chain of chain_info
(* bool = "use regexp for filenames" *) 
| Replace_Action_Text of (string list) * string * string * bool
| Replace_Trigger_Text of string * string * string * bool
| Chain3 of chain3_info

and chain_info = {
  entry_file : string;
  entry_label : string;
  dialogue : (string * Dlg.tlk_string) list ; (* SPEAKER + WORDS *)
  exit_file : string;
  exit_label : string;
} 

and chain3_info = {
  mutable c3_entry_condition : string option;
  mutable c3_entry_weight : Dlg.weight;
  c3_entry_file : string;
  c3_entry_label : string;
  c3_dialogue : c3_dialogue_unit list ; 
  mutable c3_exit_trans : Dlg.transition array ; 
  c3_variable : string option ; 
  c3_keep_first_do_with_first_speaker : bool ; 
  (* if 'true', do what Rastor suggests below: *) 
  (* Rastor: Currently, I_C_T carries over both the DO actions as well as
  the GOTO/EXTERN/REPLY commands.  Unfortunately, this can cause some
  problems if you want to interject into the ends of dialogues where an NPC
  leaves (causing your new character to EscapeArea() instead) as well as a
  few other issues that you can view here:
  http://forums.pocketplane.net/index.php?topic=16835.0.

  I propose an INTERJECT_COPY_TRANS2 command which would be identical to
  I_C_T in format, but would only carry over the GOTO/EXTERN/REPLYs in the
  original dialogue but not carry over the DOs. *)
} 

and c3_dialogue_unit = {
  c3du_speaker : string ;
  c3du_condition : string option ;
  mutable c3du_action : string option; 
  c3du_say : Dlg.tlk_string ;
  c3du_id : int ; 
} 

let c3du_counter = ref 0 

let get_c3du_counter () = 
  let ans = !c3du_counter in
  incr c3du_counter ;
  ans 

let strings_to_add = Queue.create () (* LSEs stored in order-of-creation *)
let strings_to_add_ht = Hashtbl.create 32767 
let cur_index = ref 0 

let chain_counter = ref 0 

let chain_label () = 
  incr chain_counter ; 
  Printf.sprintf "!chain_%d" !chain_counter

let available_dlgs = Hashtbl.create 31  
let d_action_list = ref ([] : (string * action) list) 

let trans_strings = ref [ Hashtbl.create 511 ]

let add_trans_strings (tsl : (int * Dlg.tlk_string) list ) =
  List.iter (fun (id,lse) -> Hashtbl.add (List.hd !trans_strings) id lse) tsl 

let push_trans () = 
  trans_strings := (Hashtbl.create 511) :: !trans_strings 

let push_copy_trans () = 
  let now = match !trans_strings with
    hd :: tl -> hd
  | [] -> Hashtbl.create 511
  in 
  trans_strings := (Hashtbl.copy now) :: !trans_strings 

let pop_trans () =
  trans_strings := List.tl !trans_strings 

let rec resolve_tlk_string game ts =
  match ts with
    Dlg.Local_String(lse) ->
      let index = 
        Stats.time "find local string" (fun () -> 
        try
          Tlk.find_string_fast lse game.Load.dialog game.Load.dialogf 
            game.Load.dialog_search
        with
          Not_found -> 
            Stats.time "strings to add" (fun () -> 
            if Hashtbl.mem strings_to_add_ht lse then begin
              Hashtbl.find strings_to_add_ht lse 
            end else begin 
              let index = !cur_index in 
              Queue.add (lse) strings_to_add ; 
              Hashtbl.add strings_to_add_ht lse index ; 
              incr cur_index ;
              index
            end ) () 
        ) () in 
        Dlg.TLK_Index(index) 
    | Dlg.Trans_String(idx) -> begin
        try
          let new_ts = Hashtbl.find (List.hd !trans_strings) idx in
          resolve_tlk_string game new_ts
        with Not_found ->
          log_and_print "ERROR: No translation provided for @%d\n" idx ;
          raise Not_found
        end
    | _ -> ts

let ok_to_resolve_strings_while_loading = ref None 
let doing_traify = ref false 
let resolve_string_while_loading ts = 
  match !ok_to_resolve_strings_while_loading with
    Some(game) -> resolve_tlk_string game ts 
  | None -> ts 

let rec single_string_of_tlk_string game ts = 
  match ts with
    Dlg.Local_String(lse) -> lse.lse_male 
  | Dlg.Trans_String(idx) -> begin
      try
        let new_ts = Hashtbl.find (List.hd !trans_strings) idx in
        single_string_of_tlk_string game new_ts
      with Not_found ->
        log_and_print "ERROR: No translation provided for @%d\n" idx ;
        raise Not_found
      end
  | Dlg.TLK_Index(idx) -> Tlk.pretty_print game.Load.dialog idx 

let set_string (g : Load.game) (i :int) (ts : Dlg.tlk_string) 
    (allow_strref : bool) : unit = 
  let rec process ts = match ts with
    Dlg.TLK_Index(i) -> 
      if (allow_strref) then begin
        if (i < 0 || i > Array.length g.Load.dialog) then begin
          log_and_print "SET_STRING %d out of range 0 -- %d\n"
            i (Array.length g.Load.dialog) ;
        end ; 
        ((g.Load.dialog.(i)) ,
        (match g.Load.dialogf with
         | None -> g.Load.dialog.(i)
         | Some(t) -> t.(i)))
      end else failwith "SET_STRING does not allow #strrefs"
  | Dlg.Local_String(lse) -> Tlk.lse_to_tlk_string lse
  | Dlg.Trans_String(idx) -> 
      begin 
      try 
        let new_lse = Hashtbl.find (List.hd !trans_strings) idx in
        process new_lse 
      with Not_found -> 
        log_and_print "ERROR: No translation provided for @%d\n" idx ;
        raise Not_found
      end 
  in
  let m, f = process ts in 
  if (i < 0 || i > Array.length g.Load.dialog) then begin
    log_and_print "SET_STRING %d out of range 0 -- %d\n"
      i (Array.length g.Load.dialog) ;
    failwith "SET_STRING out of range" 
  end ;
  log_or_print "SET_STRING #%d to %s\n" i (Tlk.short_print m 18);
  (match g.Load.dialogf with
    Some(a) -> g.Load.str_sets <- (i,g.Load.dialog.(i),a.(i)) :: 
                                  g.Load.str_sets ; 
               a.(i) <- f ; 
               g.Load.dialogf_mod <- true
  | None -> g.Load.str_sets <- (i,g.Load.dialog.(i),g.Load.dialog.(i))
                                  :: g.Load.str_sets ; 
  ) ;
  g.Load.dialog.(i) <- m ; 
  g.Load.dialog_mod <- true ;
  ()
    
let set_string_while_loading forced_index ts = 
  match !ok_to_resolve_strings_while_loading with
    Some(game) -> set_string game forced_index ts true
  | None -> failwith "cannot set strings at this point: no game loaded" 


let resolve_tlk_string_opt game tso = match tso with
    Some(ts) -> Some(resolve_tlk_string game ts)
  | None -> None

let test_trans out game = 
  let max = Array.length game.Load.dialog in
  Hashtbl.iter (fun id tlk_string ->
    let ref = match resolve_tlk_string game tlk_string with
      Dlg.TLK_Index(idx) -> idx
    | _ -> 999999
    in 
    if (ref < max) then begin
      Printf.fprintf out "@%d is #%d\n" id ref 
    end
  ) (List.hd !trans_strings)


exception FoundState of int 

let resolve_label (file,label) =
  try
    let dlg = 
      try Hashtbl.find available_dlgs file 
      with _ -> begin
        let index = 
        try 
          int_of_string label
        with _ -> 
          log_and_print "ERROR: Cannot resolve external symbolic label [%s] for DLG [%s]\n" label file ; failwith "cannot resolve label" 
        in
        raise (FoundState index) 
      end
    in 
    let found_count = ref 0 in
    let found = ref 0 in 
    Array.iteri (fun i s -> 
      if s.Dlg.symbolic_label = label then begin
        incr found_count ;
        found := i; 
      end
    ) dlg.Dlg.state ;
    (if (!found_count > 1) then begin
      log_and_print "ERROR: internal label [%s] appears %d times in processed DLG [%s]\n" label !found_count file ;
      failwith "cannot resolve label" 
    end else if (!found_count = 1) then 
      raise (FoundState(!found)))
    ; 
    try 
      let answer = int_of_string label in 
      log_and_print "WARNING: internal label [%s] not found in processed DLG [%s]\n" label file ;
      answer 
    with _ -> begin
      log_and_print "ERROR: Cannot resolve internal symbolic label [%s] for DLG [%s]\nKnown labels:" label file ;
      Array.iteri (fun i s -> 
        log_and_print " %s" s.Dlg.symbolic_label 
      ) dlg.Dlg.state ;
      log_and_print "\n" ; failwith "cannot resolve label" 
    end
  with FoundState(i) -> i


let resolve_symbolic_labels d = 
  Array.iteri (fun i s ->
    Array.iteri (fun j t ->
      match t.Dlg.next with
        Dlg.Symbolic(name,label) ->
          let idx = resolve_label (name,label) in
          t.Dlg.next <- Dlg.Absolute(name,idx) 
      | _ -> ()
    ) s.Dlg.trans
  ) d.Dlg.state

let resolve_strings game d =
  Array.iteri (fun i s ->
    try  
      s.Dlg.resp_str <- resolve_tlk_string game s.Dlg.resp_str ; 
      Array.iteri (fun j t ->
        t.Dlg.trans_str <- resolve_tlk_string_opt game t.Dlg.trans_str ;
        t.Dlg.journal_str <- match t.Dlg.journal_str with
          Some(a,str) -> Some(a,resolve_tlk_string game str)
        | None -> None 
      ) s.Dlg.trans
    with e ->
      log_and_print "ERROR: cannot resolve strings in %s\n" d.Dlg.name ;
      raise e
  ) d.Dlg.state

let locate_dlg game name =
  if Hashtbl.mem available_dlgs name then
    Hashtbl.find available_dlgs name 
  else begin
    let buff, final_path = Load.load_resource "copy_trans" game true name "DLG" in
    let dlg = Dlg.load_dlg name buff in 
    dlg 
  end

let make_available for_what game name =
  if Hashtbl.mem available_dlgs name then
    ()
  else begin
    let buff, final_path = Load.load_resource for_what game true name "DLG" in
    let dlg = Dlg.load_dlg name buff in 
    Hashtbl.add available_dlgs name dlg 
  end

let clear_state () =
  d_action_list := [] ; 
  Hashtbl.clear available_dlgs 

let preprocess_action1 game a = match a with
  | Create(d) -> 
        if Hashtbl.mem available_dlgs d.Dlg.name then begin
          log_and_print "BEGIN %s when %s is already loaded/created\n" d.Dlg.name d.Dlg.name;
          log_and_print "(you should say BEGIN %s exactly once and before APPEND %s, etc.)\n" d.Dlg.name d.Dlg.name ;
          failwith "invalid D file" 
        end ; 
        log_or_print "Adding %s to internal list of available DLGs\n" d.Dlg.name ; 
        Hashtbl.add available_dlgs d.Dlg.name d
  | _ -> () 

let action_to_str a = match a with
  | Create _ -> "CREATE"
  | Append _ -> "APPEND" 
  | Extend_Bottom _ -> "EXTEND_BOTTOM"
  | Extend_Top _ -> "EXTEND_TOP"
  | Replace _ -> "REPLACE"
  | Replace_Say _ -> "REPLACE_SAY"
  | Add_Trans_Trigger _ -> "ADD_TRANS_TRIGGER"
  | Add_Trans_Action  _ -> "ADD_TRANS_ACTION"
  | Replace_State_Trigger _ -> "REPLACE_STATE_TRIGGER"
  | Add_State_Trigger _ -> "ADD_STATE_TRIGGER"
  | Set_Weight _ -> "SET_WEIGHT"
  | Chain _ -> "CHAIN"
  | Replace_Action_Text _ -> "REPLACE_ACTION_TEXT"
  | Replace_Trigger_Text _ -> "REPLACE_TRIGGER_TEXT"
  | Chain3 _ -> "CHAIN3" 

let preprocess_action2 game a = match a with
  | Create(d) -> () 

  | Append(n,_) 
  | Extend_Top(n,_,_,_) 
  | Extend_Bottom(n,_,_,_) 
  | Replace(n,_) 
  | Replace_Say(n,_,_) 
  | Add_State_Trigger(n,_,_)
  | Replace_State_Trigger(n,_,_)
  | Add_Trans_Trigger(n,_,_,_)
  | Add_Trans_Action(n,_,_,_)
  | Set_Weight(n,_,_) 
  | Replace_Trigger_Text(n,_,_,false)
      -> make_available (action_to_str a) game n
  | Replace_Action_Text(nl,_,_,false)
      -> List.iter (fun n -> make_available (action_to_str a) game n) nl 
  | Replace_Action_Text(_,_,_,true)
  | Replace_Trigger_Text(_,_,_,true) -> () 
  | Chain(ci) ->
      let s = action_to_str a in 
      make_available s game ci.entry_file ; 
      List.iter (fun (speaker,says) -> make_available s game speaker)
        ci.dialogue 
  | Chain3(ci) ->
      let s = action_to_str a in 
      make_available s game ci.c3_entry_file ; 
      List.iter (fun c3du -> make_available s game c3du.c3du_speaker)
        ci.c3_dialogue 

let append_state n state =
  let dlg = Hashtbl.find available_dlgs n in
  dlg.Dlg.state <- Array.append dlg.Dlg.state [| state |]

let rec process_action game a =  match a with
  | Create(d) -> () 

  | Set_Weight(n,s,w) ->
    let dlg = Hashtbl.find available_dlgs n in
    let num = resolve_label (n,s) in
    dlg.Dlg.state.(num).Dlg.state_trigger_weight <- Dlg.Offset(w)

  | Replace_Action_Text(nl,s_from,s_to,use_regexp) ->
    let r = Str.regexp s_from in 
    let process dlg = 
      Array.iter (fun state ->
        Array.iter (fun trans ->
          match trans.Dlg.action with
            Some(trans_str) -> trans.Dlg.action <- 
              Some(Str.global_replace r s_to trans_str)
          | None -> () 
        ) state.Dlg.trans
      ) dlg.Dlg.state
    in 
    if use_regexp then begin
      let files_in_chitin = Key.list_of_key_resources game.Load.key in 
      let regexp_list = List.map (fun n -> Str.regexp_case_fold n) nl in 
      let matches = ref [] in
      List.iter (fun poss ->
        let b,e = split (String.uppercase poss) in
        if e = "DLG" && 
          (List.exists (fun regexp -> Str.string_match regexp b 0) 
                       regexp_list) then begin 
          make_available (action_to_str a) game b ; 
          matches := b :: !matches
        end 
      ) files_in_chitin ;
      List.iter (fun n -> process (Hashtbl.find available_dlgs n)) !matches 
    end else 
      List.iter (fun n -> 
        let dlg = Hashtbl.find available_dlgs n in
        process dlg 
      ) nl 

  | Replace_Trigger_Text(n,s_from,s_to,use_regexp) ->
    let process dlg = 
      let r = Str.regexp s_from in 
      Array.iter (fun state ->
        state.Dlg.state_trigger <- Str.global_replace r s_to 
          state.Dlg.state_trigger ;
        Array.iter (fun trans ->
          match trans.Dlg.trans_trigger with
            Some(trans_str) -> trans.Dlg.trans_trigger <- 
              Some(Str.global_replace r s_to trans_str)
          | None -> () 
        ) state.Dlg.trans
      ) dlg.Dlg.state
    in 
    if (use_regexp) then begin
      let files_in_chitin = Key.list_of_key_resources game.Load.key in 
      let regexp = Str.regexp_case_fold n in 
      let matches = ref [] in
      List.iter (fun poss ->
        let b,e = split (String.uppercase poss) in
        if e = "DLG" && Str.string_match regexp b 0 then begin 
          make_available (action_to_str a) game b ; 
          matches := b :: !matches
        end 
      ) files_in_chitin ;
      List.iter (fun n -> process (Hashtbl.find available_dlgs n)) !matches 
    end else process (Hashtbl.find available_dlgs n)


  | Append(n,sl) -> let dlg = Hashtbl.find available_dlgs n in
        dlg.Dlg.state <- Array.append dlg.Dlg.state 
         (Array.of_list sl) 
  | Extend_Top(n,sl,0,tl) -> 
      let dlg = Hashtbl.find available_dlgs n in
      List.iter (fun s -> 
        let num = resolve_label (n,s) in
        if (num >= 0 && num < Array.length dlg.Dlg.state) then
        dlg.Dlg.state.(num).Dlg.trans <- Array.append
          (Array.of_list tl) dlg.Dlg.state.(num).Dlg.trans 
        else
            log_or_print 
              "WARNING: EXTEND_TOP state #%d out of range 0-%d, SKIPPED\n"
              num (Array.length dlg.Dlg.state)
      ) sl 

  | Extend_Top(n,sl,place,tl) -> 
      let dlg = Hashtbl.find available_dlgs n in
      List.iter (fun s -> 
        let num = resolve_label (n,s) in
        if (num >= 0 && num < Array.length dlg.Dlg.state) then begin
          let s = dlg.Dlg.state.(num) in 
          let tlen = Array.length s.Dlg.trans in 
          let place = 
            if place < 0 then 
              failwith "EXTEND_TOP #position must be non-negative"
            else if place >= tlen then begin
              log_or_print 
                "WARNING: EXTEND_TOP #position %d out of range 0-%d\n"
                place tlen ; 
              tlen - 1
            end else place
          in 
          (* insert these transitions just after already-extans transition
           * #place *)
          let before = Array.sub s.Dlg.trans 0 place in
          let after = Array.sub s.Dlg.trans (place) (tlen - place) in
          let result = Array.concat [ before ; Array.of_list tl ; after ] in  
          dlg.Dlg.state.(num).Dlg.trans <- result 
        end else
            log_or_print 
             "WARNING: EXTEND_TOP state #%d out of range 0-%d, SKIPPED\n"
              num (Array.length dlg.Dlg.state)
      ) sl 

  | Extend_Bottom(n,sl,0,tl) -> 
      let dlg = Hashtbl.find available_dlgs n in
      List.iter (fun s -> 
        let num = resolve_label (n,s) in
        if (num >= 0 && num < Array.length dlg.Dlg.state) then
          dlg.Dlg.state.(num).Dlg.trans <- Array.append
            dlg.Dlg.state.(num).Dlg.trans (Array.of_list tl) 
        else
            log_or_print 
              "WARNING: EXTEND_BOTTOM state #%d out of range 0-%d, SKIPPED\n"
              num (Array.length dlg.Dlg.state)
      ) sl 

  | Extend_Bottom(n,sl,place,tl) -> 
      let dlg = Hashtbl.find available_dlgs n in
      List.iter (fun s-> 
        let num = resolve_label (n,s) in
        if (num >= 0 && num < Array.length dlg.Dlg.state) then begin
          let s = dlg.Dlg.state.(num) in 
          let tlen = Array.length s.Dlg.trans in 
          let place = 
            if place < 0 then 
              failwith "EXTEND_BOTTOM #position must be non-negative"
            else if place >= tlen then begin
              log_or_print 
                "WARNING: EXTEND_BOTTOM #position %d out of range 0-%d\n"
                place tlen ; 
              tlen - 1
            end else place
          in 
          let place = tlen - place in 
          (* insert these transitions just before already-extant transition
           * #place *)
          let before = Array.sub s.Dlg.trans 0 place in
          let after = Array.sub s.Dlg.trans (place) (tlen - place) in
          let result = Array.concat [ before ; Array.of_list tl ; after ] in  
          dlg.Dlg.state.(num).Dlg.trans <- result 
        end else
            log_or_print 
             "WARNING: EXTEND_BOTTOM state #%d out of range 0-%d, SKIPPED\n"
              num (Array.length dlg.Dlg.state)
      ) sl 

  | Replace_Say(n,l,s) -> 
    let dlg = Hashtbl.find available_dlgs n in
    let num = resolve_label (n,l) in
    dlg.Dlg.state.(num).Dlg.resp_str <- s

  | Replace(n,new_s_list) -> 
      let dlg = Hashtbl.find available_dlgs n in
      let trigger_ord = ref 0 in
      let trigger_unord = ref 0 in
      Array.iteri (fun i s -> 
        if (s.Dlg.state_trigger <> "") then begin
          ( match s.Dlg.state_trigger_weight, !trigger_ord with
            Dlg.Offset(a),b -> if (a != b) then incr trigger_unord
          | _,_ -> ());
          incr trigger_ord;
        end) dlg.Dlg.state ;      
      List.iter (fun new_s -> 
          let num = resolve_label (n,new_s.Dlg.symbolic_label) in
        if (num >= 0 && num < Array.length dlg.Dlg.state) then begin
          let old_wgt = dlg.Dlg.state.(num).Dlg.state_trigger_weight in
          let new_wgt = new_s.Dlg.state_trigger_weight in begin
            ( match !trigger_unord, old_wgt, new_wgt with
              0,_,Dlg.Not_Specified -> ()
            | 0,Dlg.Offset(a),Dlg.Offset(b) -> log_or_print "WARNING: REPLACE ignoring specified WEIGHT for state %d (%d).  DLG has trivial weights, using weight from DLG (%d).  Use SET_WEIGHT if you want to change state weights.\n" num b a
            | c,Dlg.Offset(a),Dlg.Not_Specified -> log_or_print "WARNING: REPLACE specifies no WEIGHT for state %d and DLG uses non-trivial weights.  Using weight from DLG (%d). [%d]\n" num a c
            | c,Dlg.Offset(a),Dlg.Offset(b) -> if (a != b) then log_or_print "WARNING: REPLACE ignoring specified WEIGHT for state %d (%d).  Using weight from DLG (%d).  Use SET_WEIGHT if you want to change state weights. [%d]\n" num b a c
            | _,_,_ -> ());
            dlg.Dlg.state.(num) <- new_s;
            dlg.Dlg.state.(num).Dlg.state_trigger_weight <- old_wgt
          end
        end
          else
            log_or_print "WARNING: REPLACE %d out of range 0-%d\n"
              num (Array.length dlg.Dlg.state)
        ) new_s_list 
  | Replace_State_Trigger(n,sl,new_t) -> 
        let dlg = Hashtbl.find available_dlgs n in
        List.iter (fun s -> 
        let num = resolve_label (n,s) in
        dlg.Dlg.state.(num).Dlg.state_trigger <- new_t 
        ) sl 
  | Add_State_Trigger(n,sl,new_t) -> 
        let dlg = Hashtbl.find available_dlgs n in
        List.iter (fun s -> 
        let num = resolve_label (n,s) in
        dlg.Dlg.state.(num).Dlg.state_trigger <- new_t ^ "\r\n" ^ 
        dlg.Dlg.state.(num).Dlg.state_trigger 
        ) sl 

  | Add_Trans_Action(n,state_labels,transition_indices,action_text) ->
        let dlg = Hashtbl.find available_dlgs n in
        List.iter (fun state_label -> 
          let num = resolve_label (n,state_label) in
          let do_it trans = 
          match trans.Dlg.action with
            None -> trans.Dlg.action <- Some(action_text)
          | Some(str) -> trans.Dlg.action <- Some(action_text ^ "\r\n" ^ str)
          in 
          match transition_indices with
          | [] -> (* do them all *) 
            Array.iter do_it dlg.Dlg.state.(num).Dlg.trans 
          | lst -> 
            List.iter (fun i ->
              if (i >= 0 && 
                  i < Array.length dlg.Dlg.state.(num).Dlg.trans) then begin
                let trans = dlg.Dlg.state.(num).Dlg.trans.(i) in 
                do_it trans 
              end 
            ) lst 
        ) state_labels 

  | Add_Trans_Trigger(n,sl,new_t,t_list) -> 
        let dlg = Hashtbl.find available_dlgs n in
        List.iter (fun s -> 
        let num = resolve_label (n,s) in
        let do_it trans = 
        match trans.Dlg.trans_trigger with
          None -> trans.Dlg.trans_trigger <- Some(new_t )
        | Some(str) -> trans.Dlg.trans_trigger <- Some(new_t ^ "\r\n" ^ str)
        in 
        match t_list with
        | [] -> 
          Array.iter do_it dlg.Dlg.state.(num).Dlg.trans 
        | tl -> 
          List.iter (fun trans_idx ->
            let i = my_int_of_string trans_idx in 
            let trans = dlg.Dlg.state.(num).Dlg.trans.(i) in 
            do_it trans 
          ) tl 
        ) sl 
  | Chain(ci) -> begin
      let rec process_chain cl in_label = match cl with
        (f1,s1) :: (f2,s2) :: tl ->
          let new_label = chain_label () in 
          let new_state = Dlg.make_state s1 in_label f2 new_label in
          append_state f1 new_state ;
          process_chain ((f2,s2)::tl) new_label 
      | (f1,s1) :: [] -> let new_state = Dlg.make_state s1 
            in_label ci.exit_file ci.exit_label in
          append_state f1 new_state 
      | [] -> ()
      in 
      process_chain ci.dialogue ci.entry_label 
    end

  | Chain3(ci) -> begin
      let label_of_ci = Hashtbl.create 255 in
      List.iter (fun c3du -> 
        Hashtbl.add label_of_ci c3du (c3du.c3du_speaker,(chain_label ()))
      ) ci.c3_dialogue ;
      (match ci.c3_variable with
      | None -> Hashtbl.replace label_of_ci (List.hd ci.c3_dialogue) 
          (ci.c3_entry_file,ci.c3_entry_label) ;
      | Some _ -> ()
      ); 
      let final_trans = Array.to_list ci.c3_exit_trans in 
      let keep = ci.c3_keep_first_do_with_first_speaker in 
      (*
      log_or_print "DEBUG: I_C_T: %b (%d)\n" keep 
        (List.length final_trans); 
        *)
      let first_actions, final_trans = match final_trans, keep with 
      | hd :: tl ,true -> 
        begin 
        let orig_act = hd.Dlg.action in 
        let rec process lst = match lst with
        | [] -> () 
        | hd' :: tl' -> 
          if (hd'.Dlg.action <> orig_act) then begin
            let ps s_opt = match s_opt with
            | Some(s) -> s
            | None -> "\"\"" 
            in 
            log_or_print "WARNING: I_C_T2: the interjection point (%s %s) has multiple exit transitions that have different actions!\nFirst Action: %s\nOther Action: %s\n" ci.c3_entry_file ci.c3_entry_label
            (ps hd'.Dlg.action) (ps orig_act)
          end ; 
          hd'.Dlg.action <- None ;
          process tl' 
        in 
        process (hd :: tl) ;
        orig_act, (hd :: tl) 
        end ;
      | x,false -> None, x
      | _,true -> failwith "ERROR: CHAIN3: 'keep first do with first speaker' requires that you have at least one exit transition"
      in 

      let combine_some s1 s2 = match s1, s2 with
      | Some(s1), Some(s2) -> Some(s1 ^ "\n" ^ s2)
      | Some(s1), None -> Some(s1)
      | None,Some(s2) -> Some(s2)
      | None,None -> None
      in 
        
      let rec possible_trans action cl = match cl with
        | [] -> 
          List.rev (
            List.map (fun t -> 
              { t with Dlg.action = combine_some t.Dlg.action action }
            ) final_trans
          ) (* will be reversed again inside process_chain *) 
        | ci :: tl -> begin
            let a,b = Hashtbl.find label_of_ci ci in
            let trans = { Dlg.unknown_flags = 0;
                          Dlg.trans_str = None ;
                          Dlg.journal_str = None ;
                          Dlg.trans_trigger = ci.c3du_condition ;
                          Dlg.action = action ;
                          Dlg.next = Dlg.Symbolic(a,b) } in
            trans :: (match ci.c3du_condition with 
              | None -> [] 
              | Some _ -> (possible_trans action tl))
            end 
      in 
      let rec process_chain cl condition weight = match cl with
        | ci :: tl -> 
          let (this_file,this_label) = Hashtbl.find label_of_ci ci in
          let trans = List.rev (possible_trans ci.c3du_action tl) in 
          let trans = Array.of_list trans in 
          let new_state = Dlg.make_state_trans ci.c3du_say this_label trans in
          (match condition with
            Some(s) -> new_state.Dlg.state_trigger <- s;
                       new_state.Dlg.state_trigger_weight <- weight ;
          | None -> ()) ; 
          append_state ci.c3du_speaker new_state ; 
          process_chain tl None (Dlg.Not_Specified)
        | [] -> () 
      in 
      process_chain ci.c3_dialogue ci.c3_entry_condition ci.c3_entry_weight;
      begin 
      match ci.c3_variable with
      | None -> ()
      | Some(var) -> 
          begin
          let fst = List.hd ci.c3_dialogue in
          let file,label = Hashtbl.find label_of_ci fst in  
          let trans = Dlg.make_trans_of_next (Dlg.Symbolic(file,label)) in
          trans.Dlg.trans_trigger <- Some 
            (Printf.sprintf "Global(\"%s\",\"GLOBAL\",0)%s" var 
            (match fst.c3du_condition with Some(s) -> (" " ^ s) | None -> ""));
          let a1 = Some
            (Printf.sprintf "SetGlobal(\"%s\",\"GLOBAL\",1)" var) in
          trans.Dlg.action <- combine_some a1 first_actions ; 
          process_action game (Extend_Bottom(ci.c3_entry_file,
            [ci.c3_entry_label],0,[trans]))
          end 
      end  
    end


let pctta game tl = (* process_copy_trans__trans_array *)
  (* given a transition, return the list of transition it expands to *)
  try
  let process t = 
    match t.Dlg.next with
      Dlg.Copy(f,s) -> 
        let d = locate_dlg game f in 
        let i = resolve_label (f,s) in
        if i < 0 || i >= Array.length d.Dlg.state then begin
          log_and_print 
            "ERROR: COPY_TRANS %s state #%d out of range 0-%d, SKIPPED\n"
            f i (Array.length d.Dlg.state) ;
          failwith "COPY_TRANS out of range" 
        end ; 
        let lst = Array.to_list (d.Dlg.state.(i).Dlg.trans) in
        List.map Dlg.duplicate_trans lst 
    | _ -> [t] 
  in 
  let expanded_trans_a = Array.map process tl in 
  let expanded_trans_list = Array.to_list expanded_trans_a in
  let expanded_trans_list = List.flatten expanded_trans_list in
  let expanded_trans_a = Array.of_list expanded_trans_list in
  expanded_trans_a 
  with s ->
    log_and_print "ERROR: Cannot process COPY_TRANS\n" ; 
    raise s 

let process_copy_trans game a = match a with 
  | Create(d) -> 
      Array.iter (fun s -> 
        s.Dlg.trans <- pctta game s.Dlg.trans
      ) d.Dlg.state ;
      Create(d) 
  | Append(s,sl) -> 
      List.iter (fun s -> s.Dlg.trans <- pctta game s.Dlg.trans) sl ;
      Append(s,sl)
  | Extend_Top(a,b,pos,tl) ->    Extend_Top(a,b,pos,
    Array.to_list (pctta game (Array.of_list tl)))
  | Extend_Bottom(a,b,pos,tl) -> Extend_Bottom(a,b,pos,
    Array.to_list (pctta game (Array.of_list tl)))
  | Replace(n,sl) -> 
      List.iter (fun s -> s.Dlg.trans <- pctta game s.Dlg.trans) sl ;
      Replace(n,sl)
  | Replace_Say(_,_,_)
  | Add_State_Trigger(_,_,_) 
  | Replace_State_Trigger(_,_,_) 
  | Add_Trans_Trigger(_,_,_,_) 
  | Add_Trans_Action(_,_,_,_) 
  | Set_Weight(_,_,_) 
  | Replace_Action_Text(_,_,_,_)
  | Replace_Trigger_Text(_,_,_,_)
  | Chain(_) -> a
  | Chain3(c) -> c.c3_exit_trans <- pctta game c.c3_exit_trans ; a


let postprocess_dlg game name d = 
            (*      try  *)
                  Stats.time "resolve strings" (resolve_strings game) d ;
                  Stats.time "resolve labels" (resolve_symbolic_labels) d
                  (* with s ->
                    log_and_print "ERROR: Cannot postprocess %s\n" name ; 
                    raise s *)

let dc game lst = 
  let what = ref "" in 
  let where = ref "" in 
  try 
    what := "preprocessing CREATE" ;
    Stats.time "preprocess .D actions" 
      (List.iter (fun (w,a) -> where := w; preprocess_action1 game a)) lst ;
    what := "preprocessing" ;
    Stats.time "preprocess .D actions" 
      (List.iter (fun (w,a) -> where := w;  preprocess_action2 game a)) lst ;
    what := "processing COPY_TRANS" ; 
    let lst = Stats.time "process copy_trans" 
      List.map (fun (w,a) -> where := w ; (w,process_copy_trans game a)) lst in 
    what := "processing .D actions" ;
    Stats.time "process .D actions" 
      (List.iter (fun (w,a) -> where := w; process_action game a)) lst ;
    what := "postprocessing" ; 
    Stats.time "postprocess .D actions" 
      (Hashtbl.iter (fun n d -> where := n; postprocess_dlg game n d))
      available_dlgs ;
    ()
  with e -> 
    log_and_print "ERROR: %s [%s]: %s\n" !what !where 
      (Printexc.to_string e);
    raise e

