(* Diff routines using xdiff library *)
open Util
open Xdiff
open Str

(* Usage: new_buff, bad_chunks, app_chunks = do_patch orig_buff patch_buff 
   
   Applies patch_buff to orig_buff to create new_buff.  The number
   of chunks that couldn't be applied is returned in bad_chunks.
   The number of chunks that are already applied is returned in app_chunks.
 *)
let do_patch orig_buff patch_buff vb = begin
  let newf, rejf = Xdiff.patch orig_buff patch_buff in begin
    (* RE which deliminates rejected chunks from patch *)
    let delimre = Str.regexp "@@ -[0-9]+,[0-9]+ \\+[0-9]+,[0-9]+ @@" in
    let rej_chunks = Str.full_split delimre rejf in
    let bad_chunks = ref 0 in
    let app_chunks = ref 0 in
    let rec check_chunk = function
        [] -> ()
      | Delim(stra) :: Text(strb) :: rem -> 
          let chunk = stra ^ strb in
          let newff, rejff = Xdiff.revpatch orig_buff chunk in 
          let begre = Str.regexp "^" in
          let chunk = Str.global_replace begre "    " chunk in
          let lfre = Str.regexp_string "\r" in
          let chunk = Str.global_replace lfre "" chunk in
          let _ = 
            if (rejff <> "") then begin
              incr bad_chunks;
              log_or_print "!!! FAILED PATCH CHUNK:\n";
              if vb then log_or_print "%s\n" chunk;
            end
            else begin
              incr app_chunks;
              log_or_print "*** CHUNK ALREADY APPLIED:\n";
              if vb then log_or_print "%s\n" chunk;
            end
          in
          check_chunk rem;
      | _ -> (log_and_print "do_patch: chunk from Xdiff.diff has wrong format.";
              raise Not_found)
    in
    check_chunk rej_chunks;
    (newf, !bad_chunks, !app_chunks) ;
  end
end

(* Usage: create_patch orig_buff new_buff out_name ncont
   
   Creates a patch by comparing orig_buff and new_buff and
   writes the patch to file out_name.
 *)
let create_patch orig_buff new_buff out_name ncont = begin
  try
    let out_chan = open_for_writing out_name true in
    let patch_buff = Xdiff.diff orig_buff new_buff ncont in
    let _ = output_string out_chan patch_buff in
    close_out out_chan
  with e -> begin
    log_and_print "create_patch: couldn't create patch: %s\n" (Printexc.to_string e);
    raise Not_found
  end
end

