%{
open Util
open Load

(*** input handle ***)

let current_unit = ref None

(*** Error handling ***)
let parse_error = Util.parse_error 

let get_current_unit () = match !current_unit with
  Some(s) -> s
| None -> parse_error "No current unit (use BEGIN)" 

type trans_opt_type =
    Trans_Reply of Dlg.tlk_string option
  | Trans_Do of string option
  | Trans_Journal of (Dlg.journal_type * Dlg.tlk_string) option
  | Trans_Flags of int

let extra_actions = ref [] 

let verify_trigger_list s =  
  let con = the_context () in 
  let filename = (Printf.sprintf "trigger list near line %d, column %d of %s"
    con.line con.col con.filename) in 
  let lexbuf = lex_init_from_internal_string filename s in  
  let lexbuf = Lexing.from_string (String.copy s) in 
  let result = try 
    let res = Bafparser.trigger_list Baflexer.initial lexbuf in  
    let buff = Buffer.create (String.length s) in 
    Bcs.print_script_text (the_game()) (Bcs.Save_BCS_Buffer(buff))
        (Bcs.BCS_Print_TriggerList(res)) false None ;
    Buffer.contents buff 
  with e -> 
    log_only "WARNING: cannot verify trigger ~%s~: %s\n" s 
      (Printexc.to_string e) ;
    s 
  in 
  pop_context () ; 
  result  

let verify_action_list s =   
  let con = the_context () in 
  let filename = (Printf.sprintf "action list near line %d, column %d of %s"
    con.line con.col con.filename) in 
  let lexbuf = lex_init_from_internal_string filename s in 
  let result = try 
    let res = Bafparser.action_list Baflexer.initial lexbuf in 
    let buff = Buffer.create (String.length s) in 
    Bcs.print_script_text (the_game()) (Bcs.Save_BCS_Buffer(buff))
        (Bcs.BCS_Print_ActionList(res)) false None ;
    Buffer.contents buff 
  with e -> 
    log_only "WARNING: cannot verify action ~%s~: %s\n" s 
      (Printexc.to_string e) ;
    s 
  in
  pop_context () ; 
  (*
  (if result = "" then begin
    let warn = con.warn_only in
    con.warn_only <- true ;
    (try parse_error "Empty Action (may confuse some utilities, like NI)"
    with _ -> () ) ;
    con.warn_only <- warn 
  end) ; 
  *)
  result  

%}

%token QUESTION COLON SEMICOLON
%token ACTION_IF
%token PATCH_IF
%token ADD_CRE_ITEM
%token ADD_KIT
%token ADD_KNOWN_SPELL
%token ADD_MAP_NOTE
%token ADD_MUSIC
%token ADD_PROJECTILE
%token ADD_STATE_TRIGGER
%token ADD_TRANS_TRIGGER
%token ADD_TRANS_ACTION
%token ADD_STORE_ITEM
%token ALLOW_MISSING
%token ALWAYS
%token AND OR NOT ELSE LPAREN RPAREN
%token APPEND
%token APPENDI
%token APPEND_COL
%token ASK_EVERY_COMPONENT
%token AT_EXIT
%token AT_INTERACTIVE_EXIT
%token AT_INTERACTIVE_UNINSTALL
%token SET_2DA_ENTRY
%token READ_2DA_ENTRY
%token AT_UNINSTALL
%token AUTHOR
%token AUTO_TRA
%token BACKUP
%token BEGIN
%token BUT_ONLY_IF_IT_CHANGES
%token CHAIN2
%token CHAIN3
%token COMPILE
%token COMPILE_BAF_TO_BCS
%token COMPILE_D_TO_DLG
%token DECOMPILE_BCS_TO_BAF
%token DECOMPILE_DLG_TO_D
%token COPY
%token GLOB
%token EQUIP
%token TWOHANDED
%token COPY_EXISTING
%token COPY_EXISTING_REGEXP
%token COPY_RANDOM
%token COPY_TRANS
%token DELETE_BYTES
%token DEPRECATED
%token DESIGNATED
%token DO
%token END
%token EQUALS
%token EQUALSEQUALS
%token EXIT
%token EXTEND_BOTTOM
%token EXTEND_BOTTOM_REGEXP
%token EXTEND_TOP
%token EXTEND_TOP_REGEXP
%token EXTERN
%token FAIL 
%token FILE_CONTAINS
%token FILE_MD5
%token FILE_EXISTS
%token FILE_EXISTS_IN_GAME
%token FILE_SIZE
%token FLAGS
%token FORBID_FILE
%token FOR 
%token GOTO
%token GT GTE LT LTE
%token BAND BOR BXOR
%token BLSL BASR BLSR BNOT
%token IF
%token SET
%token IF_EVAL
%token IF_SIZE_IS
%token INSERT_BYTES
%token INSERT_FILE
%token INTERJECT
%token INTERJECT_COPY_TRANS
%token INTERJECT_COPY_TRANS2
%token JOURNAL SOLVED_JOURNAL UNSOLVED_JOURNAL
%token LANGUAGE
%token MENU_STYLE
%token MKDIR
%token NO_LOG_RECORD
%token PATCH_GAM
%token PLUS MINUS TIMES DIVIDE
%token PRINT 
%token RANDOM_SEED
%token RANDOM
%token READ_ASCII
%token READ_BYTE
%token READ_LONG
%token READ_SHORT
%token REMOVE_KNOWN_SPELL
%token REPLACE
%token REPLACE_ACTION_TEXT
%token REPLACE_ACTION_TEXT_PROCESS
%token REPLACE_ACTION_TEXT_REGEXP
%token REPLACE_ACTION_TEXT_PROCESS_REGEXP
%token REPLACE_BCS_BLOCK
%token REPLACE_BCS_BLOCK_REGEXP
%token APPLY_BCS_PATCH
%token APPLY_BCS_PATCH_OR_COPY
%token REPLACE_SAY
%token REPLACE_STATE_TRIGGER
%token REPLACE_TEXTUALLY
%token REPLACE_EVALUATE
%token REPLACE_TRIGGER_TEXT
%token REPLACE_TRIGGER_TEXT_REGEXP
%token REPLY
%token REQUIRE_FILE
%token REQUIRE_PREDICATE
%token REQUIRE_COMPONENT
%token FORBID_COMPONENT
%token SAY
%token SUBCOMPONENT
%token SCRIPT_STYLE
%token SET_WEIGHT
%token STRING
%token STRING_CONCAT
%token STRING_SET
%token STRING_EQUAL
%token STRING_EQUAL_CASE
%token THEN
%token UNINSTALL
%token UNLESS
%token USING
%token WEIGHT
%token WRITE_ASCII
%token WRITE_EVALUATED_ASCII
%token WRITE_BYTE
%token WRITE_FILE
%token WRITE_LONG
%token WRITE_SHORT
%token WHILE

%token EOF

%token <string> SOUND STRING
%token <string * string> INLINED_FILE
%token <int> STRING_REF TRANS_REF FORCED_STRING_REF

%left AND
%left OR
%left PLUS MINUS
%left TIMES DIVIDE
%left QUESTION COLON
%left GT GTE LT LTE EQUALS
%nonassoc NOT

/* Non-terminals informations */
%start d_file tra_file tp_file log_file tutu_file

%type <(string * int * int * (string option)) list> log_file installed_mod_list
%type <(string * int)> begin_prologue
%type <(string)> append_prologue replace_prologue
%type <(bool * string * (string list) * int)> extend_prologue
%type <(string * string * string * bool)> interject_prologue interject_copy_trans_prologue
%type <Dc.action list> d_file action_list
%type <Dc.action> action
%type <Dlg.state list> state_list state
%type <Dlg.transition list> transition_list
%type <Dlg.transition> transition

%type <Dlg.weight> weight

%type <(string * Dlg.tlk_string) list> chain2_list

%type <((string option) * Dlg.tlk_string * (string option)) list> chain3_list
%type <string option> optional_condition

%type <Dlg.tlk_string> lse
%type <Dlg.tlk_string list> say_list
%type <string> sound_opt
%type <Dlg.trans_next> next

%type <trans_opt_type> trans_opt
%type <trans_opt_type list> trans_opt_list

%type <(int * Dlg.tlk_string) list> tra_file
%type <(int * Dlg.tlk_string) list> tutu_file

%type <Tp.tp_file> tp_file
%type <Tp.tp_lang list> tp_lang_list
%type <Tp.tp_flag list> tp_flag_list
%type <Tp.tp_mod list> tp_mod_list
%type <Tp.tp_action list> tp_action_list
%type <Tp.tp_action> tp_action
%type <Tp.tp_constraint list> tp_when_list
%type <(string * string) list> str_str_list
%type <Tp.tp_patch list> tp_patch_list
%type <string list> string_list
%type <int list> int_list
%type <string list> upper_string_list
%type <int> hash_int_option

%type <string option * string> appendi_prologue

%%

d_file : action_list   { let answer = $1 @ !extra_actions in 
                         extra_actions := [] ;
                         answer } 
;

action_list :           { [] }
| action action_list    { $1 :: $2 } 
; 

begin_prologue : 
  BEGIN STRING             { current_unit := Some(String.uppercase $2); (String.uppercase $2,0) } 
| BEGIN STRING STRING       { current_unit := Some(String.uppercase $2); (String.uppercase $2,my_int_of_string $3) } 
; 

append_prologue : APPEND STRING       
      { current_unit := Some(String.uppercase $2) ; 
        (String.uppercase $2) } ; 

extend_prologue : 
  EXTEND_TOP STRING string_list hash_int_option 
    { current_unit := Some(String.uppercase $2) ; 
    (true,String.uppercase $2,$3,$4) } 
| EXTEND_BOTTOM STRING string_list hash_int_option
    { current_unit := Some(String.uppercase $2) ;
    (false,String.uppercase $2,$3,$4) }
; 

hash_int_option :             { 0 }
| STRING_REF                  { $1 } 
;

replace_prologue : REPLACE STRING 
    { current_unit := Some(String.uppercase $2); (String.uppercase $2) } 
; 
replace_state_trigger_prologue : REPLACE_STATE_TRIGGER STRING STRING STRING 
  string_list
  { current_unit := Some(String.uppercase $2); 
    (String.uppercase $2,$3 :: $5,$4) }
; 

opt_do_string_list :    { [] } 
| DO string_list        { $2 } 
; 

add_trans_trigger_prologue : ADD_TRANS_TRIGGER STRING STRING STRING string_list
opt_do_string_list 
  { current_unit := Some(String.uppercase $2); 
    (String.uppercase $2,$3 :: $5,$4,$6) }
; 
add_state_trigger_prologue : ADD_STATE_TRIGGER STRING STRING STRING string_list
  { current_unit := Some(String.uppercase $2); 
    (String.uppercase $2,$3 :: $5,$4) 
  }
; 

chain3_prologue : CHAIN3 optional_weighted_condition STRING STRING 
  { current_unit := Some(String.uppercase $3); ($2,String.uppercase $3,$4) } 
;

interject_prologue : INTERJECT STRING STRING STRING
  { current_unit := Some(String.uppercase $2); (String.uppercase $2,$3,$4,false) } 
;

interject_copy_trans_prologue : INTERJECT_COPY_TRANS STRING STRING STRING
  { current_unit := Some(String.uppercase $2); (String.uppercase $2,$3,$4,false) } 
| INTERJECT_COPY_TRANS2 STRING STRING STRING 
  { current_unit := Some(String.uppercase $2); (String.uppercase $2,$3,$4,true) } 

;

action : 
begin_prologue state_list { let name,flags = $1 in Dc.Create(
  { Dlg.name = name ;
    Dlg.state = Array.of_list $2 ;
    Dlg.dlg_flags = flags ; })}
| append_prologue state_list END 
  { current_unit := None ; Dc.Append( $1 , $2 ) } 
| extend_prologue transition_list END 
  { let top,ext_unit,ext_label,number = $1 in
    current_unit := None ; 
    if top then 
      Dc.Extend_Top( String.uppercase ext_unit, ext_label, number, $2 ) 
    else
      Dc.Extend_Bottom( String.uppercase ext_unit, ext_label, number, $2 ) 
  } 
| replace_prologue state_list END 
  { current_unit := None ; 
    Dc.Replace($1, $2) } 
| REPLACE_SAY STRING STRING lse 
  { Dc.Replace_Say($2,$3,$4) } 
| replace_state_trigger_prologue
  { let f,s,t = $1 in 
    let verified_t = verify_trigger_list t in 
    current_unit := None ; 
    Dc.Replace_State_Trigger(f,s,verified_t) } 
| add_state_trigger_prologue 
  { let f,s,t = $1 in 
    let verified_t = verify_trigger_list t in 
    current_unit := None ; 
    Dc.Add_State_Trigger(f,s,verified_t) } 
| add_trans_trigger_prologue 
  { let f,s,t,tl = $1 in 
    let verified_t = verify_trigger_list t in 
    current_unit := None ; 
    Dc.Add_Trans_Trigger(f,s,verified_t,tl) } 
| ADD_TRANS_ACTION STRING BEGIN string_list END BEGIN int_list END STRING
  { current_unit := Some(String.uppercase $2);
    let verified_a = verify_action_list $9 in 
    current_unit := None ; 
    Dc.Add_Trans_Action($2,$4,$7,$9) }
| SET_WEIGHT STRING STRING STRING_REF
  { Dc.Set_Weight($2,$3,$4) }
| chain3_prologue chain3_list compound_chain3_list chain3_epilogue
  { let (entry_weight,entry_cond),file,state = $1 in 
    let first_part = List.map (fun (cond,says,action) -> 
      {
        Dc.c3du_speaker = file ;
        Dc.c3du_condition = cond ;
        Dc.c3du_action = action ;
        Dc.c3du_say = says ;
        Dc.c3du_id = Dc.get_c3du_counter () ; 
      } ) $2 in 
    Dc.Chain3
    {
      Dc.c3_entry_condition = entry_cond;
      Dc.c3_entry_weight = entry_weight;
      Dc.c3_entry_file = file ;
      Dc.c3_entry_label = state ;
      Dc.c3_dialogue = first_part @ $3 ;
      Dc.c3_variable = None ; 
      Dc.c3_exit_trans = $4 ; 
      Dc.c3_keep_first_do_with_first_speaker = false; 
    } 
  } 
| interject_prologue compound_chain3_list chain3_epilogue
  { let file,label,var,keep_do = $1 in 
    Dc.Chain3
    {
      Dc.c3_entry_condition = None;
      Dc.c3_entry_weight = Dlg.Not_Specified;
      Dc.c3_entry_file = file ;
      Dc.c3_entry_label = label ;
      Dc.c3_dialogue = $2 ;
      Dc.c3_variable = Some(var) ; 
      Dc.c3_exit_trans = $3 ;
      Dc.c3_keep_first_do_with_first_speaker = keep_do; 
    } 
  } 
| interject_copy_trans_prologue compound_chain3_list END
  { let file,label,var,keep_do = $1 in 
    Dc.Chain3
    {
      Dc.c3_entry_condition = None;
      Dc.c3_entry_weight = Dlg.Not_Specified;
      Dc.c3_entry_file = file ;
      Dc.c3_entry_label = label ;
      Dc.c3_dialogue = $2 ;
      Dc.c3_variable = Some(var) ; 
      Dc.c3_exit_trans = [| Dlg.make_trans_of_next (Dlg.Copy(file,label)) |] ;
      Dc.c3_keep_first_do_with_first_speaker = keep_do; 
    } 
  } 
| REPLACE_TRIGGER_TEXT STRING STRING STRING 
  { Dc.Replace_Trigger_Text(String.uppercase $2,$3,$4,false) } 
| REPLACE_TRIGGER_TEXT_REGEXP STRING STRING STRING 
  { Dc.Replace_Trigger_Text(String.uppercase $2,$3,$4,true) } 
| REPLACE_ACTION_TEXT STRING STRING STRING upper_string_list
  { Dc.Replace_Action_Text(String.uppercase $2 :: $5,$3,$4,false) } 
| REPLACE_ACTION_TEXT_PROCESS STRING STRING STRING upper_string_list
  { Dc.Replace_Action_Text(String.uppercase $2 :: $5,$3,verify_action_list $4,false) } 
| REPLACE_ACTION_TEXT_REGEXP STRING STRING STRING upper_string_list
  { Dc.Replace_Action_Text(String.uppercase $2 :: $5,$3,$4,true) } 
| REPLACE_ACTION_TEXT_PROCESS_REGEXP STRING STRING STRING upper_string_list
  { Dc.Replace_Action_Text(String.uppercase $2 :: $5,$3,verify_action_list $4,true) } 
; 

chain3_list : optional_condition lse optional_action     { [($1,$2,$3)] }
| optional_condition lse optional_action EQUALS chain3_list 
  { ($1,$2,$3) :: (List.map (fun (a,b,c) -> 
    match $1,a with
    | _,None -> ($1,b,c)
    | None,_ -> (a,b,c)
    | Some(first),Some(second) -> (Some(first ^ "\n" ^ second),b,c)
    ) $5) }
;

optional_action :       { None }
| DO STRING             { let verified_action = verify_action_list $2 in 
                          Some(verified_action) } 
; 

chain3_epilogue: END STRING STRING   
    { let next = Dlg.Symbolic(String.uppercase $2,$3) in
      [| Dlg.make_trans_of_next next |] }
| EXTERN STRING STRING               
    { let next = Dlg.Symbolic(String.uppercase $2,$3) in
      [| Dlg.make_trans_of_next next |] }
| COPY_TRANS STRING STRING           
    { let next = Dlg.Copy(String.uppercase $2,$3) in 
      [| Dlg.make_trans_of_next next |] }
| EXIT                               
    { let next = Dlg.Exit in 
      [| Dlg.make_trans_of_next next |] }
| END transition_list                { Array.of_list $2 } 
; 

optional_weighted_condition :    { ((Dlg.Not_Specified),(None)) }
| IF weight STRING THEN { ($2,(Some(verify_trigger_list $3))) } 
; 

optional_condition :    
| IF STRING optional_then  { Some(verify_trigger_list $2) } 
| { None }
; 

compound_chain3_list :                  { [] } 
| EQUALSEQUALS STRING chain3_list compound_chain3_list 
  { let this_speaker = String.uppercase $2 in 
    let first_part = List.map (fun (cond,says,action) -> 
      { Dc.c3du_speaker = this_speaker ;
        Dc.c3du_condition = cond ;
        Dc.c3du_action = action ;
        Dc.c3du_say = says ; 
        Dc.c3du_id = Dc.get_c3du_counter () ; 
        }) $3 in 
    first_part @ $4 
  } 
;



state_list :            { [] }
| state state_list      { $1 @ $2 } 
;

weight :                { Dlg.Not_Specified }
| WEIGHT STRING_REF     { Dlg.Offset($2) }
;

optional_then: { () }
| THEN { () }
; 
optional_begin: { () }
| BEGIN { () }
; 

state_trigger : 
STRING { if $1 = "" then "" else verify_trigger_list $1 } 
; 

state : 
IF weight state_trigger optional_then optional_begin STRING 
  SAY say_list
  transition_list
END {
      let state_trigger = $3 in 
      if List.length $8 = 1 then 
         [{ Dlg.resp_str = List.hd $8 ;
           Dlg.trans = Array.of_list $9 ; 
           Dlg.state_trigger = state_trigger ;
           Dlg.state_trigger_weight = $2 ; 
           Dlg.symbolic_label = $6 ; }]
      else begin
        let rec process_say_list sl in_label in_trig in_weight = match sl with
          s1 :: s2 :: tl -> 
            let new_label = Dc.chain_label () in
            let dest_file = get_current_unit () in 
            let new_state = Dlg.make_state s1 in_label dest_file new_label in
            new_state.Dlg.state_trigger_weight <- in_weight ; 
            new_state.Dlg.state_trigger <- in_trig ; 
            new_state :: (process_say_list (s2::tl) new_label "" Dlg.Not_Specified)
        | s1 :: [] -> [
          let state = Dlg.make_state_trans s1 in_label (Array.of_list $9)
          in state.Dlg.state_trigger_weight <- in_weight ;
          state ] 
        | [] -> [] 
        in 
        process_say_list $8 $6 $3 $2
      end
    } 
/* backwards compat: internal append */
| appendi_prologue state_list END
  {
    let old_unit, append_name = $1 in 
    extra_actions := (Dc.Append( append_name, $2 )) :: !extra_actions ;
    current_unit := old_unit ;
    [] 
  }
/* backwards compat: internal 2-person chain */
| CHAIN2 STRING STRING chain2_list END STRING STRING 
  { let rec convert cl extern_guy = match cl with
      [] -> []
    | (code,s) :: tl -> 
        let file = if extern_guy then $2 else get_current_unit () in
        let extern_guy = if code = "==" then not extern_guy else extern_guy in
        (file,s) :: convert tl extern_guy
    in 
    extra_actions := (Dc.Chain(
      { Dc.entry_file = String.uppercase $2 ;
        Dc.entry_label = $3 ;
        Dc.dialogue = convert $4 true;
        Dc.exit_file = String.uppercase $6 ;
        Dc.exit_label = $7 ;
      } )) :: !extra_actions ;
    []  
  } 
;

appendi_prologue : APPENDI STRING 
 { let what = String.uppercase $2 in  
  let old = !current_unit in current_unit := Some(what); (old, what) } 
; 

chain2_list : lse               { [("",$1)] } 
| lse EQUALS chain2_list        { ("",$1) :: $3 }
| lse EQUALSEQUALS chain2_list  { ("==",$1) :: $3 }
; 

say_list : lse                  { [$1] } 
| lse EQUALS say_list           { $1 :: $3 } 
; 

transition_list :            { [] }
| transition transition_list { $1 :: $2 } 
;

trans_opt :
| REPLY lse      { Trans_Reply(Some($2)) } 
| DO STRING      { let verified_action = verify_action_list $2 in 
                    Trans_Do(Some(verified_action)) }
| JOURNAL lse    { Trans_Journal(Some(Dlg.Normal_Journal,$2)) }
| SOLVED_JOURNAL lse    { Trans_Journal(Some(Dlg.Solved_Journal,$2)) }
| UNSOLVED_JOURNAL lse    { Trans_Journal(Some(Dlg.Unsolved_Journal,$2)) }
| FLAGS STRING   { Trans_Flags(my_int_of_string $2)}
; 

trans_opt_list :                { [] }
| trans_opt trans_opt_list      { $1 :: $2 } 
;

trans_trigger : STRING { if $1 = "" then None else
  Some(verify_trigger_list $1) } 
; 

optional_string : STRING { $1 }
| { "" }
; 

transition :
IF trans_trigger optional_then trans_opt_list next 
  { let result = { 
      Dlg.trans_trigger = $2 ; 
      Dlg.trans_str = None ;
      Dlg.action = None ;
      Dlg.journal_str = None;
      Dlg.unknown_flags = 0 ;
      Dlg.next = $5 ; 
    } in
    List.iter (fun elt -> match elt with
    | Trans_Reply(r) -> result.Dlg.trans_str <- r ;
    | Trans_Do(r) -> result.Dlg.action <- r ;
    | Trans_Journal(r) -> 
        (if (result.Dlg.journal_str <> None) then 
         try parse_error "You may only have one JOURNAL per transition.  Recovering." with _ -> ()) ;
        result.Dlg.journal_str <- r;
    | Trans_Flags(r) -> result.Dlg.unknown_flags <- r;
    ) $4 ;
    result 
  } 
| PLUS optional_string PLUS lse trans_opt_list next 
  { let result = { 
      Dlg.trans_trigger = 
        if $2 = "" then None
        else Some(verify_trigger_list $2) ; 
      Dlg.trans_str = Some($4) ;
      Dlg.action = None ;
      Dlg.journal_str = None;
      Dlg.unknown_flags = 0 ;
      Dlg.next = $6 ; 
    } in
    List.iter (fun elt -> match elt with
    | Trans_Reply(r) -> result.Dlg.trans_str <- r ;
    | Trans_Do(r) -> result.Dlg.action <- r ;
    | Trans_Journal(r) -> result.Dlg.journal_str <- r;
    | Trans_Flags(r) -> result.Dlg.unknown_flags <- r;
    ) $5 ;
    result 
  }
| COPY_TRANS STRING STRING
  { { 
      Dlg.trans_trigger = None ; 
      Dlg.trans_str = None ;
      Dlg.action = None ;
      Dlg.journal_str = None;
      Dlg.unknown_flags = 0 ;
      Dlg.next = Dlg.Copy($2,$3) ;
    } 
  } 
;

next : GOTO STRING     { Dlg.Symbolic(get_current_unit () ,$2) }
| PLUS STRING     { Dlg.Symbolic(get_current_unit () ,$2) }
| EXTERN STRING STRING { Dlg.Symbolic(String.uppercase $2,$3) } 
| EXIT                 { Dlg.Exit } 
; 

lse_string:     STRING { $1 }
| STRING STRING_CONCAT lse_string { $1 ^ $3 } 
; 

lse : lse_string sound_opt  
  { let result = Dlg.Local_String({ lse_male = $1 ; lse_male_sound = $2; 
                    lse_female = $1; lse_female_sound = $2;}) in 
    (match !Dlg.local_string_ht with
      Some(l) -> if not (List.mem result l ) then begin
          Dlg.local_string_ht := Some(result :: l )
        end 
    | _ -> () ) ;
    result
  }
| lse_string sound_opt lse_string sound_opt 
  { let result = Dlg.Local_String({ lse_male = $1 ; lse_male_sound = $2; 
          lse_female = $3; lse_female_sound = $4; }) in 
    (match !Dlg.local_string_ht with
      Some(l) -> if not (List.mem result l) then 
        Dlg.local_string_ht := Some(result :: l)
    | _ -> () ) ;
    result
  }
| STRING_REF { Dlg.TLK_Index($1) } 
| TRANS_REF  { Dc.resolve_string_while_loading (Dlg.Trans_String($1)) } 
| FORCED_STRING_REF lse 
  { let _ = Dc.set_string_while_loading $1 $2 in Dlg.TLK_Index($1) } 
;

sound_opt :             { "" }
| SOUND                 { $1 } 
; 

tra_file :              { [] }
| TRANS_REF EQUALS lse tra_file { ($1,$3) :: $4 } 
;

tutu_file :              { [] }
| STRING_REF EQUALS lse tutu_file { ($1,$3) :: $4 }
;

tp_file : 
BACKUP STRING AUTHOR STRING tp_flag_list tp_lang_list tp_mod_list 
  { { Tp.backup = $2;
      Tp.author = $4; 
      Tp.flags = $5 ;
      Tp.languages = $6 ;
      Tp.module_list = $7 ;
  } }
; 

tp_flag_list :                  { [] } 
| AUTO_TRA STRING tp_flag_list         { (Tp.Auto_Tra($2)) :: $3 } 
| MENU_STYLE STRING tp_flag_list       { (Tp.Menu_Style($2)) :: $3 } 
| ASK_EVERY_COMPONENT tp_flag_list     { (Tp.Ask_Every_Component) :: $2 } 
| ALWAYS tp_action_list END tp_flag_list   { (Tp.Always($2)) :: $4 } 
| ALLOW_MISSING upper_string_list tp_flag_list
    { Tp.Allow_Missing($2) :: $3 }
| SCRIPT_STYLE STRING tp_flag_list
    { let n = match (String.uppercase $2) with
      | "BG" 
      | "IWD" 
      | "BG2" -> Load.BG
      | "PST" -> Load.PST
      | "IWD2" -> Load.IWD2
      | _ -> parse_error "unknown SCRIPT_STYLE" 
      in 
      Tp.Script_Style(n) :: $3 } 
; 

upper_string_list :            { [] }
| STRING upper_string_list  { (String.uppercase $1) :: $2 } 
; 

tp_lang_list :   { [] } 
| LANGUAGE STRING STRING string_list tp_lang_list
  { { Tp.lang_name = $2 ;
      Tp.lang_dir_name = $3 ;
      Tp.lang_tra_files = $4 ;
  } :: $5 } 
;

tp_mod_list :         { [] } 
| BEGIN lse 
  tp_mod_flag_list 
  tp_action_list tp_mod_list 
  { { Tp.mod_name = $2 ; Tp.mod_parts = $4 ; 
      Tp.mod_flags = $3 } :: $5 }
; 

tp_mod_flag_list :    { [] }
| SUBCOMPONENT lse tp_mod_flag_list 
  { Tp.TPM_SubComponents($2,Tp.Pred_True) :: $3 } 
| NO_LOG_RECORD tp_mod_flag_list { Tp.TPM_NotInLog :: $2 } 
| SUBCOMPONENT lse tp_predicate tp_mod_flag_list 
  { Tp.TPM_SubComponents($2,$3) :: $4 } 
| DEPRECATED lse tp_mod_flag_list { Tp.TPM_Deprecated($2) :: $3 }
| DESIGNATED STRING tp_mod_flag_list { Tp.TPM_Designated(my_int_of_string $2) :: $3 }
| REQUIRE_COMPONENT STRING STRING lse tp_mod_flag_list 
  { Tp.TPM_RequireComponent($2,(my_int_of_string $3),$4) :: $5 } 
| FORBID_COMPONENT STRING STRING lse tp_mod_flag_list 
  { Tp.TPM_ForbidComponent($2,(my_int_of_string $3),$4) :: $5 } 
| REQUIRE_PREDICATE tp_predicate lse tp_mod_flag_list { Tp.TPM_RequirePredicate($2,$3) :: $4 }
; 

tp_action_list :            { [] }
| tp_action tp_action_list  { $1 :: $2 }
; 

optional_using :                { [] } 
| USING string_list             { $2 }
; 

optional_plus :     { true }
| PLUS              { false } 
; 

optional_equip :    { false }
| EQUIP             { true }
;

optional_2h :       { true }
| TWOHANDED           { false }
;

optional_glob :     { false }
| GLOB              { true }
; 

tp_action : 
| COPY optional_plus optional_glob str_str_list tp_patch_list tp_when_list 
  { Tp.TP_Copy(
      { Tp.copy_get_existing = false;
        Tp.copy_use_regexp = false; 
        Tp.copy_use_glob = $3; 
        Tp.copy_file_list = $4 ;
        Tp.copy_patch_list = $5 ;
        Tp.copy_constraint_list = $6;
        Tp.copy_backup = $2 ; } ) }
| COPY_EXISTING optional_plus str_str_list tp_patch_list tp_when_list 
  { Tp.TP_Copy(
      { Tp.copy_get_existing = true;
        Tp.copy_use_regexp = false; 
        Tp.copy_use_glob = false;
        Tp.copy_file_list = $3 ;
        Tp.copy_patch_list = $4 ;
        Tp.copy_constraint_list = $5;
        Tp.copy_backup = $2 ; } ) }
| COPY_EXISTING_REGEXP optional_plus optional_glob str_str_list tp_patch_list tp_when_list 
  { Tp.TP_Copy(
      { Tp.copy_get_existing = true;
        Tp.copy_use_regexp = true; 
        Tp.copy_use_glob = $3; 
        Tp.copy_file_list = $4 ;
        Tp.copy_patch_list = $5 ;
        Tp.copy_constraint_list = $6;
        Tp.copy_backup = $2 ; } ) }
| COPY_RANDOM string_list tp_patch_list { Tp.TP_CopyRandom($2,$3) } 
| RANDOM_SEED STRING { Tp.TP_RandomSeed(my_int_of_string $2) } 
| COMPILE string_list optional_using { Tp.TP_Compile($2,$3) }
| INLINED_FILE { Tp.TP_Inlined_File($1) } 
| MKDIR string_list { Tp.TP_Mkdir($2) } 
| REQUIRE_FILE STRING lse { Tp.TP_Require_File($2,$3) } 
| FORBID_FILE STRING lse { Tp.TP_Forbid_File($2,$3) } 
| APPEND STRING STRING tp_when_list { Tp.TP_Append($2,$3,$4) } 
| APPEND_COL STRING STRING tp_when_list 
    { let lst = Str.split (Str.regexp "[ \t]+") $3 in
      let lst = List.map (fun elt -> if elt = "$" then "" else elt) lst in
      Tp.TP_Append_Col($2,lst,$4) } 
| EXTEND_TOP STRING STRING tp_patch_list optional_using
    { Tp.TP_Extend_Top(false,$2,$3,$4,$5) } 
| EXTEND_BOTTOM STRING STRING tp_patch_list optional_using
    { Tp.TP_Extend_Bottom(false,$2,$3,$4,$5) } 
| EXTEND_TOP_REGEXP STRING STRING tp_patch_list optional_using
    { Tp.TP_Extend_Top(true,$2,$3,$4,$5) } 
| EXTEND_BOTTOM_REGEXP STRING STRING tp_patch_list optional_using
    { Tp.TP_Extend_Bottom(true,$2,$3,$4,$5) } 
| AT_EXIT STRING { Tp.TP_At_Exit($2) } 
| AT_INTERACTIVE_EXIT STRING { Tp.TP_At_Interactive_Exit($2) } 
| AT_INTERACTIVE_UNINSTALL STRING { Tp.TP_At_Interactive_Uninstall($2) } 
| AT_UNINSTALL STRING { Tp.TP_At_Uninstall($2) } 
| ADD_MUSIC STRING STRING { Tp.TP_Add_Music(
    { Tp.music_name = $2; 
      Tp.music_file = $3; } ) } 
| ADD_PROJECTILE STRING { 
  Tp.TP_Add_Projectile( { Tp.pro_file = $2; } ) }
| STRING_SET STRING lse { Tp.TP_String_Set($2,$3,None) }  
| STRING_SET STRING lse USING STRING { Tp.TP_String_Set($2,$3,Some($5)) }  
| UNINSTALL STRING STRING { Tp.TP_Uninstall_Now($2,(my_int_of_string $3)) } 
| ADD_KIT 
    STRING 
    STRING 
    STRING
    STRING
    STRING 
    STRING 
    STRING 
    STRING 
    STRING 
    STRING 
    STRING 
    STRING
    STRING
    STRING
    SAY lse 
    SAY lse 
    SAY lse 
  { Tp.TP_Add_Kit(
   { Tp.kit_name = $2 ;
    Tp.clasweap = $3 ; 
    Tp.weapprof = $4 ;
    Tp.abclasrq = $5 ;
    Tp.abclsmod = $6 ;
    Tp.abdcdsrq = $7 ;
    Tp.abdcscrq = $8 ;
    Tp.alignmnt = $9 ;
    Tp.dualclas = $10 ;
    Tp.ability_file = $11 ;
    Tp.include_in = $12 ;
    Tp.unused_class = $13 ;
    Tp.tob_abbr = $14; 
    Tp.tob_start = 
      (let lst = Str.split (Str.regexp "[ \t]+") $15 in
      List.map (fun elt -> if elt = "$" then "" else elt) lst) ;
    Tp.lower = $17 ;
    Tp.mixed = $19 ;
    Tp.help = $21 ;
  })}
| FAIL lse       { Tp.TP_Fail($2) }  
| PRINT lse       { Tp.TP_Print($2) }  
| ACTION_IF tp_predicate THEN BEGIN tp_action_list END { Tp.TP_If($2,$5,[]) } 
| ACTION_IF tp_predicate THEN BEGIN tp_action_list END ELSE 
    BEGIN tp_action_list END { Tp.TP_If($2,$5,$9) } 
;

tp_predicate : LPAREN tp_predicate RPAREN { $2 }  
| tp_predicate AND tp_predicate { Tp.Pred_And($1,$3) }
| tp_predicate OR tp_predicate { Tp.Pred_Or($1,$3) }
| NOT tp_predicate { Tp.Pred_Not($2) }
| LPAREN patch_exp RPAREN { Tp.Pred_Expr($2) } 
| FILE_EXISTS STRING { Tp.Pred_File_Exists($2) }
| FILE_MD5 STRING STRING { Tp.Pred_File_MD5($2,$3) }
| FILE_EXISTS_IN_GAME STRING { Tp.Pred_File_Exists_In_Game($2) }
| FILE_SIZE STRING STRING { Tp.Pred_File_Size($2,my_int_of_string $3) }
| FILE_CONTAINS STRING STRING { Tp.Pred_File_Contains($2,$3) }
;


string_list :            { [] }
| STRING string_list  { $1 :: $2 } 
; 

int_list:   { [] }
| STRING int_list  { (my_int_of_string $1) :: $2 }
;

tp_when_list :                 { [] }
| IF STRING tp_when_list       { Tp.TP_Contains($2) :: $3 }
| UNLESS STRING tp_when_list   { Tp.TP_NotContains($2) :: $3 } 
| IF_SIZE_IS STRING tp_when_list { Tp.TP_IfSizeIs(my_int_of_string $2) :: $3 }
| IF_EVAL patch_exp tp_when_list { Tp.TP_Eval($2) :: $3 }
| BUT_ONLY_IF_IT_CHANGES tp_when_list        { Tp.TP_ButOnlyIfItChanges :: $2 } 
; 

str_str_list :               { [] }
| STRING STRING str_str_list { ($1,$2) :: $3 } 
; 

patch_exp : STRING              { Tp.PE_String($1) }
| LPAREN patch_exp RPAREN       { $2 } 
| STRING STRING_EQUAL STRING    { Tp.PE_StringEqual($1,$3,false) } 
| STRING STRING_EQUAL_CASE STRING    { Tp.PE_StringEqual($1,$3,true) } 
| RANDOM LPAREN patch_exp patch_exp RPAREN { Tp.PE_Random($3,$4) } 
| patch_exp PLUS patch_exp      { Tp.PE_Add($1,$3) } 
| patch_exp MINUS patch_exp      { Tp.PE_Sub($1,$3) } 
| patch_exp TIMES patch_exp      { Tp.PE_Mul($1,$3) } 
| patch_exp DIVIDE patch_exp      { Tp.PE_Div($1,$3) } 
| patch_exp EQUALS patch_exp  { Tp.PE_Equal($1,$3) }  
| NOT patch_exp               { Tp.PE_Not($2) } 
| patch_exp AND patch_exp  { Tp.PE_And($1,$3) } 
| patch_exp OR patch_exp  { Tp.PE_Or($1,$3) } 
| patch_exp GT patch_exp  { Tp.PE_GT($1,$3) } 
| patch_exp GTE patch_exp  { Tp.PE_GTE($1,$3) } 
| patch_exp LT patch_exp  { Tp.PE_LT($1,$3) } 
| patch_exp LTE patch_exp  { Tp.PE_LTE($1,$3) } 

| patch_exp BAND patch_exp  { Tp.PE_BAND($1,$3) } 
| patch_exp BOR patch_exp  { Tp.PE_BOR($1,$3) } 
| BNOT patch_exp  { Tp.PE_BNOT($2) } 
| patch_exp BXOR patch_exp  { Tp.PE_BXOR($1,$3) } 
| patch_exp BLSL patch_exp  { Tp.PE_BLSL($1,$3) } 
| patch_exp BLSR patch_exp  { Tp.PE_BLSR($1,$3) } 
| patch_exp BASR patch_exp  { Tp.PE_BASR($1,$3) } 

| patch_exp QUESTION patch_exp COLON patch_exp { Tp.PE_If($1,$3,$5); } 
;

tp_patch_list :   { [] }
| tp_patch tp_patch_list { $1 :: $2 }
;

tp_patch   : 
| SAY patch_exp lse { Tp.TP_PatchStrRef($2,$3) }
| REPLACE STRING lse { Tp.TP_PatchString($2,$3) }
| REPLACE_TEXTUALLY STRING STRING { Tp.TP_PatchStringTextually($2,$3) }
| REPLACE_EVALUATE STRING BEGIN tp_patch_list END STRING { Tp.TP_PatchStringEvaluate($2,$4,$6) }
| REPLACE_BCS_BLOCK STRING STRING { Tp.TP_PatchReplaceBCSBlock($2,$3) } 
| REPLACE_BCS_BLOCK_REGEXP STRING STRING { Tp.TP_PatchReplaceBCSBlockRE($2,$3) } 
| APPLY_BCS_PATCH STRING { Tp.TP_PatchApplyBCSPatch($2,None) }
| APPLY_BCS_PATCH_OR_COPY STRING STRING { Tp.TP_PatchApplyBCSPatch($2,Some($3)) }
| READ_ASCII patch_exp STRING { Tp.TP_PatchReadAscii($2,$3) }
| READ_BYTE patch_exp STRING { Tp.TP_PatchReadByte($2,$3) } 
| READ_SHORT patch_exp STRING { Tp.TP_PatchReadShort($2,$3) } 
| READ_LONG patch_exp STRING { Tp.TP_PatchReadLong($2,$3) } 
| WRITE_FILE patch_exp STRING { Tp.TP_PatchWriteFile($2,$3,false) }  
| INSERT_FILE patch_exp STRING { Tp.TP_PatchWriteFile($2,$3,true) }  
| WRITE_BYTE patch_exp patch_exp { Tp.TP_PatchByte($2,$3) } 
| WRITE_SHORT patch_exp patch_exp { Tp.TP_PatchShort($2,$3) } 
| WRITE_LONG patch_exp patch_exp { Tp.TP_PatchLong($2,$3) } 
| WRITE_ASCII patch_exp STRING { Tp.TP_PatchASCII($2,$3,false) } 
| WRITE_EVALUATED_ASCII patch_exp STRING { Tp.TP_PatchASCII($2,$3,true) } 
| INSERT_BYTES patch_exp patch_exp { Tp.TP_PatchInsertBytes($2,$3) }
| DELETE_BYTES patch_exp patch_exp { Tp.TP_PatchDeleteBytes($2,$3) }
| SET STRING EQUALS patch_exp { Tp.TP_PatchSet($2,$4) } 
| STRING EQUALS patch_exp { Tp.TP_PatchSet($1,$3) } 
| ADD_KNOWN_SPELL STRING STRING_REF STRING { Tp.TP_Add_Known_Spell($2,$3,$4) }
| WHILE patch_exp BEGIN tp_patch_list END { Tp.TP_PatchWhile($2,$4) }
| FOR LPAREN tp_patch_list SEMICOLON patch_exp SEMICOLON tp_patch_list 
    RPAREN BEGIN tp_patch_list END { Tp.TP_PatchFor($3,$5,$7,$10) } 
| PATCH_IF patch_exp optional_then BEGIN tp_patch_list END { Tp.TP_PatchIf($2,$5,[]) }
| PATCH_IF patch_exp optional_then BEGIN tp_patch_list END ELSE BEGIN tp_patch_list END 
  { Tp.TP_PatchIf($2,$5,$9) }
| SET_2DA_ENTRY patch_exp patch_exp patch_exp patch_exp { Tp.TP_Patch2DA($2,$3,$4,$5) }
| ADD_MAP_NOTE STRING_REF STRING_REF STRING lse 
  { Tp.TP_Add_Map_Note(
   {Tp.xcoord = $2 ;
    Tp.ycoord = $3 ;
    Tp.mstring = $5 ;
    Tp.colour = $4 ; })}
| READ_2DA_ENTRY patch_exp patch_exp patch_exp STRING { Tp.TP_Read2DA($2,$3,$4,$5) }
| ADD_CRE_ITEM 
   STRING
   STRING_REF 
   STRING_REF 
   STRING_REF 
   STRING 
   STRING 
   optional_equip
   optional_2h
  { Tp.TP_Add_Cre_Item(
   { Tp.item_name = $2 ;
     Tp.i_charge1 = $3 ;
     Tp.i_charge2 = $4 ;
     Tp.i_charge3 = $5 ;
     Tp.i_flags = $6 ;
     Tp.item_slot = $7 ;
     Tp.equip = $8 ;
     Tp.twohanded_weapon = $9 ;})}
| PATCH_GAM STRING STRING STRING_REF STRING_REF { Tp.TP_Patch_Gam($2,$3,$4,$5) }
| ADD_STORE_ITEM optional_plus STRING STRING_REF STRING_REF STRING_REF STRING STRING_REF
  { Tp.TP_Add_S_Item({ Tp.overwrite_store_item = $2; },$3,$4,$5,$6,$7,$8) }
| REMOVE_KNOWN_SPELL upper_string_list { Tp.TP_Remove_Known_Spell($2) }
| COMPILE_BAF_TO_BCS { Tp.TP_CompileBAFtoBCS } 
| DECOMPILE_BCS_TO_BAF { Tp.TP_CompileBCStoBAF } 
| COMPILE_D_TO_DLG { Tp.TP_CompileDtoDLG } 
| DECOMPILE_DLG_TO_D { Tp.TP_CompileDLGtoD } 
; 

log_file : installed_mod_list { $1 }
;

installed_mod_list :          { [] }
| STRING STRING_REF STRING_REF installed_mod_list 
  { ($1,$2,$3,None) :: $4 }
| STRING STRING_REF STRING_REF STRING installed_mod_list 
  { ($1,$2,$3,Some($4)) :: $5 }
; 
  
%%



