open Util
open Xdiff
open Version 

(* big generic parsing function *) 
let parse_buffer filename buffer sort_of_file parse_lex_fun =
  try 
    begin 
    let lexbuf : Lexing.lexbuf = lex_init_from_string filename buffer in
    try 
      let result = Stats.time sort_of_file 
        (fun () -> parse_lex_fun lexbuf) () in
      pop_context () ;
      log_or_print "[%s] parsed\n" filename ; 
      result
    with e -> 
      (try input_error "" (Printexc.to_string e) with _ -> () ) ; 
      pop_context () ;
      raise e 
    end 
  with e -> 
    log_and_print "ERROR: parsing [%s]: %s\n" 
      filename (Printexc.to_string e) ; 
    raise e 

let load_log () =
  try
    let result = parse_file Tp.log_name "parsing .log files" 
      (Dparser.log_file Dlexer.initial) in 
    Tp.the_log := List.map (fun (a,b,c,d) -> ((String.uppercase a),b,c,d,Tp.Installed)) result
  with e -> 
    log_or_print "WARNING: parsing log [%s]: %s\n" Tp.log_name 
      (Printexc.to_string e) ;
    Tp.the_log := [] 

let handle_baf_filename filename =
  parse_file filename "parsing .baf files" 
    (Bafparser.baf_file Baflexer.initial) 

let compile_baf_filename game filename =
  try 
    let script = parse_file filename "parsing .baf files" 
      (Bafparser.baf_file Baflexer.initial) in 
    let name,ext = split (Filename.basename filename) in 
    let out = open_for_writing ("override/" ^ name ^ ".bcs") true in
    Bcs.save_bcs game (Bcs.Save_BCS_OC(out)) script ;
    close_out out 
  with e ->
    (log_and_print "ERROR: error compiling [%s]: %s\n" 
      filename (Printexc.to_string e) ; raise e)

let handle_script_buffer filename buffer = 
  match split (String.uppercase filename) with
  | _,"BAF" -> parse_buffer filename buffer "parsing .baf files" 
              (Bafparser.baf_file Baflexer.initial) 
  | _,_     -> parse_buffer filename buffer "parsing .bcs files" 
              (Bcsparser.bcs_file Bcslexer.initial) 

let handle_dlg_buffer game filename buffer = 
  let emit_from = !Dlg.emit_from in
  let comments = !Dlg.comments in 
  let emit_text = !Dlg.emit_text in 
  let dlg = Dlg.load_dlg filename buffer in 
  Dlg.emit_from := false; 
  Dlg.comments := false; 
  Dlg.emit_text := false; 
  let out_buff = Buffer.create (String.length buffer) in 
  Dlg.emit_d dlg filename game.Load.dialog game.Load.dialogf
    out_buff None None (fun str _ -> str) false ;
  Dlg.emit_from := emit_from; 
  Dlg.comments := comments; 
  Dlg.emit_text := emit_text; 
  Buffer.contents out_buff 

let handle_d_buffer game filename buffer =
  try 
    let result = parse_buffer filename buffer "parsing .d files"
      (Dparser.d_file Dlexer.initial) in 
    (match result with
    | [Dc.Create(dlg) as act] -> 
      Dc.dc game [(filename,act)] ; 
      let out_buff = Buffer.create (1024 * 32) in 
      Dlg.save_dlg dlg out_buff ;
      Buffer.contents out_buff 
    | _ -> failwith "COMPILE_D_TO_DLG -- must simply define a DLG") ;
  with e -> 
    Dc.clear_state () ;
    raise e


let handle_tp2_filename filename =
  parse_file filename "parsing .tp2 files" 
    (Dparser.tp_file Dlexer.initial) 

let handle_tra_filename filename =
  if file_exists filename then begin
    let result = parse_file filename "parsing .tra files" 
      (Dparser.tra_file Dlexer.initial) in 
    log_or_print "[%s] has %d translation strings\n" filename 
      (List.length result); 
    Dc.add_trans_strings result 
  end 

let handle_d_filename filename = 
  try 
    let result = parse_file filename "parsing .d files"
          (Dparser.d_file Dlexer.initial) in
    let result = List.map (fun a -> (filename,a)) result in 
    Dc.d_action_list := result @ !Dc.d_action_list 
  with e -> 
    (Dc.clear_state () ; raise e)

let reprint_d_action str pfun = begin
  let lexbuf = lex_init_from_string "printing DLG" str in 
  let result = try 
    let res = Bafparser.action_list Baflexer.initial lexbuf in 
    let buff = Buffer.create (String.length str) in 
    Bcs.print_script_text (Load.the_game()) (Bcs.Save_BCS_Buffer(buff))
        (Bcs.BCS_Print_ActionList(res)) false (Some(pfun)) ; 
    Buffer.contents buff 
  with e -> begin
    str 
  end 
  in
  pop_context () ;
  result 
end 

let emit_dlg_files game output_dir = 
  if (!Dc.d_action_list <> []) then begin
    (try 
      Stats.time "process .D files" (Dc.dc game) !Dc.d_action_list ;
    with e ->
      Dc.clear_state () ;
      (*log_and_print "ERROR: problem processing D files: %s\n"
        (Printexc.to_string e) ; *)
      raise e) ;
    Hashtbl.iter (fun name dlg ->
      let filename = output_dir ^ "/" ^ name ^ ".DLG" in
      try 
        let out_buff = Buffer.create (1024 * 32) in 
        Dlg.save_dlg dlg out_buff ;
        let out_chan = open_for_writing filename true in
        Buffer.output_buffer out_chan out_buff ;
        close_out out_chan ; 
      with e ->
        Dc.clear_state () ;
        log_and_print "ERROR: problem saving [%s]: %s\n" filename
          (Printexc.to_string e) ;
        raise e
    ) Dc.available_dlgs ;
  end ;
  Dc.clear_state () 

let pause_at_end = ref false 

let main () = 

  let user_min = ref None in 
  let user_max = ref None in 

  let cmp_src = ref None in
  let cmp_dest = ref None in 

  let make_an_itemlist = ref false in 
  let make_an_xplist = ref false in 
  let do_cre_analysis = ref false in 
  let do_itmsort = ref false in 

  let dcmp_src = ref None in
  let dcmp_dest = ref None in

  let tcmp_src = ref None in
  let tcmp_dest = ref None in

  let bcmp_src = ref None in
  let bcmp_dest = ref None in

  let bcmp_orig = ref None in
  let bcmp_patch = ref None in

  let tlkcmp_src = ref None in
  let tlkcmp_dest = ref None in

  let make_biff = ref None in 
  let remove_biff = ref None in 
  let make_tlk = ref [] in 

  let no_game = ref false in 

  let transitive = ref false in 
  let two_pass = ref false in 

  let dlg_list = ref [] in
  let d_list = ref [] in
  let dout_list = ref [] in 
  let ds_list = ref [] in 
  let strapp_list = ref [] in 
  let bc_list = ref [] in 
  let bg_list = ref [] in 
  let bcs_list = ref [] in 
  let baf_list = ref [] in 

  let no_auto_update = ref false in 
  let auto_update_all = ref false in 

  let automate_list = ref [] in 
  let automate_min = ref 62169 in 

  let tlk_merge = ref [] in 

  let extract_tlk = ref false in 
  let extract_kits = ref (0) in 

  let tp_list = ref [] in 
  let output_dir = ref "." in

  let strfind_list = ref [] in 

  let trans_list = ref [] in

  let d_headers = ref true in 

  let list_biff = ref false in 
  let list_files = ref false in 

  let list_eff_list = ref [] in 

  let textout = ref stdout in 

  let bs_type_list = ref [] in
  let bs_str_list = ref [] in 

  let biff_short = ref 0 in
  let biff_short_at = ref 0 in 

  let output_dialog = ref None in 
  let output_dialogf = ref None in 

  let traify = ref None in 
  let traify_num = ref 0 in 

  let forceify = ref None in 

  let use_trans = ref false in 
  let test_trans = ref false in 

  let argv0_base, argv0_ext = split (String.uppercase 
    (Filename.basename Sys.argv.(0))) in 

  let verify_latest can_spawn = begin

    let version_regexp = Str.regexp_case_fold ".*version \\([0-9]+\\).*" in 

    let digest_ht = Hashtbl.create 255 in 

    let this,ext = split (Filename.basename Sys.argv.(0)) in 

    let my_real_name = this^".exe" in 

    let this_digest = Digest.file my_real_name in 

    Hashtbl.add digest_ht this_digest (int_of_string version) ; 

    let weidu_list = ref [] in 

    (if not (Str.string_match (Str.regexp_case_fold "setup-.*exe") my_real_name 0) then weidu_list := (my_real_name,int_of_string version) :: !weidu_list) ;

    (try begin
      let d_h = Unix.opendir "." in
      try
        while true do 
        let f = Unix.readdir d_h in

        if Str.string_match (Str.regexp_case_fold "setup-.*exe") f 0 then begin
          let f_digest = Digest.file f in 
          let version = 
          if Hashtbl.mem digest_ht f_digest then
            Hashtbl.find digest_ht f_digest
          else begin 
            let newstdin, newstdin' = Unix.pipe () in 
            let newstdout, newstdout' = Unix.pipe () in 
            let newstderr, newstderr' = Unix.pipe () in 
            let pid = Arch.create_process_env
              f [| "WeiDU-Backup" ; "--game bar" |] [| |] newstdin newstdout' newstderr'
            in 
            log_and_print "{%s} Queried (pid = %d)" f pid ; 
            let ic = Unix.in_channel_of_descr newstdout in 
            let line = input_line ic in 
            let version = 
              try
              let s = Str.global_replace version_regexp "\\1" line in 
              int_of_string s 
              with _ -> -1
            in 
            (try Unix.close newstdin with _ -> ()) ;
            (try Unix.close newstdout with _ -> ()) ;
            (try Unix.close newstderr with _ -> ()) ;
            (try Unix.close newstdin' with _ -> ()) ;
            (try Unix.close newstdout' with _ -> ()) ;
            (try Unix.close newstderr' with _ -> ()) ;
            log_and_print " version = %d" version ; 
            let pid', ps = Unix.waitpid [] pid in 
            log_and_print " query done.\n" ; 
            Hashtbl.add digest_ht f_digest version ; 
            version 
          end in
          weidu_list := (f,version) :: !weidu_list
        end done
      with _ -> ()
     end with _ -> ()) ;

    let sorted = List.sort (fun (f1,v1) (f2,v2) -> v2 - v1) !weidu_list in 

    (* head of list is newest element *) 

    if (List.length sorted) > 1 then begin 

    let newest,newest_t = List.hd sorted in
    let oldest,oldest_t = List.hd (List.rev sorted) in

    if (newest_t <> oldest_t) then begin
      (* out-of-synch: time to do updates *)
      log_and_print "Newest WeiDU is version %d, updating!\n" newest_t ; 
      log_and_print "WeiDU files in version order:\n" ;
      List.iter (fun (f,v) -> log_and_print "  [%s] version %d\n" f v ) sorted ;

      let newest_buff = load_file newest in 

      List.iter (fun (target,target_t) ->
        if (target <> this) && (target_t <> newest_t) then begin 
        (* log_and_print "\tUnlinking [%s]: " target ;  *)
        let unlink_worked = (try Unix.unlink target ; true with _ -> false) in
        (* log_and_print "%b\n" unlink_worked ; *)
        log_and_print "\tCopying [%s] -> [%s]: " newest target ; 
        let copy_worked = try 
          let out = open_for_writing target true in
          output_string out newest_buff ;
          close_out out ; 
          true
        with _ -> false in
        log_and_print "%b\n" copy_worked ;
        end 
      ) sorted ;

      if newest_t <> (int_of_string version) then begin 
        let not_this = Filename.basename 
          (let file,time = (List.find (fun (f,v) -> f <> this) sorted) in
          file)
        in 

        Sys.argv.(0) <- not_this ;

        let cmd = Array.fold_left (fun acc elt -> acc ^ " " ^ elt)
          Sys.argv.(0) (Array.sub Sys.argv 1 ((Array.length Sys.argv)-1)) in

        let env = Array.append [|Printf.sprintf "weiduautoupdate=%s" my_real_name|]
          (Unix.environment ())
        in 

        if (can_spawn) then begin 
          let _ = Unix.execve not_this Sys.argv env in 
          exit 1 ;
        end 
      end 
    end ;
   end 
  end in 

  let auto () = begin
    pause_at_end := true ; 
    output_dialog := Some("dialog.tlk") ;
    output_dialogf := Some("dialogf.tlk") ;
    Load.set_dialog_tlk_path "dialog.tlk" ;
    Load.set_dialogf_tlk_path "dialogf.tlk" ;
    init_log (argv0_base ^ ".DEBUG") ; 
    (try
      if (Arch.do_auto_update) then begin
        if not !no_auto_update then 
          verify_latest true 
      end else 
        log_and_print "[On this architecture, WeiDU does not auto-update.\n  You must ensure that you have the most recent version.]\n"
    with e -> 
      begin
        log_and_print "ERROR: Cannot perform auto-update, exiting!\n\t%s\n"
          (Printexc.to_string e) ;
        exit 1
      end ) ;
    if List.exists (fun arg -> let a,b = split arg in (String.uppercase b) = "TP2") 
      (Array.to_list Sys.argv) then
      () (* setup-solaufein.exe foo.tp2 
          * runs foo.tp2, not setup-solaufein.tp2 *) 
    else begin 
      let rec try_it file_list = match file_list with
      | file :: lst ->  
          if file_exists file then 
            tp_list := file :: !tp_list
          else try_it lst
      | [] -> 
        log_and_print "\n\n** ERROR ** [%s.TP2] not found.\nMake sure that you have unpacked the archive correctly and\nthat you are not trying to run this file from inside an archive." argv0_base
      in
      let chunk_list = Str.split (Str.regexp "[-]") argv0_base in 
      let chunk = match chunk_list with
      | a :: b :: _ -> b 
      | _ -> ""
      in 
      try_it
      [ (argv0_base ^ ".TP2") ;
        (chunk ^ ".TP2") ;
        (chunk ^ "/" ^ argv0_base ^ ".TP2") ; 
        (chunk ^ "/" ^ chunk ^ ".TP2") ; ]
    end 
  end in 

  let forced_script_style = ref Load.BG in 

  let usageMsg = Printf.sprintf "\t\tWeiDU (version %s: \"%s\")\n\nusage: WeiDU [options] BAF,BCS,D,DLG,TRA,TP,TP2-files\n\nGeneral Input Options:\n" version comment in
  let argDescr = [
    "--game", Arg.String Load.add_game_path, "X\tset main game directory to X" ;
    "--nogame", Arg.Set no_game,"\tdo not load any default game files" ; 
    "--search", Arg.String Load.add_override_path, "X\tlook in X for input files (cumulative)" ;
    "--tlkin", Arg.String Load.set_dialog_tlk_path,"X\tuse X as DIALOG.TLK" ;
    "--ftlkin", Arg.String Load.set_dialogf_tlk_path,"X\tuse X as DIALOGF.TLK";
    "--tlkmerge", Arg.String (fun s -> tlk_merge := !tlk_merge @ [s]),
      "X\tmerge X into loaded DIALOG.TLK" ;
    "--yes", Arg.Set Tp.always_yes,"\tanswer all TP2 questions with 'Yes'"
    ;
    "--uninstall", Arg.Set Tp.always_uninstall,"\tanswer all TP2 questions with 'Uninstall'" ;
    "--noautoupdate", Arg.Set no_auto_update,"\tdo not auto-update WeiDU setup files" ;
    "--update-all", Arg.Set auto_update_all,"\tauto-update all WeiDU setup files"; 
    "--script-style", Arg.String (fun s -> 
      let n = match String.uppercase s with
        | "PST" -> Load.PST
        | "BG" -> Load.BG
        | "BG2" -> Load.BG
        | "IWD" -> Load.BG
        | "IWD2" -> Load.IWD2
        | _ -> failwith "unknown script-style" 
      in forced_script_style := n),"X\tuse BCS/BAF style X (BG, PST, IWD2)"

    ^ "\n\nGeneral Output Options:\n" ;

    "--out", Arg.String (fun s -> output_dir := s), "X\temit all output files in directory X" ;
    "--backup", Arg.String (fun s -> backup_dir := Some(s)), "X\tbackup files to directory X before overwriting" ;
    "--tlkout", Arg.String (fun s -> output_dialog := Some(s)), "X\temit X as new DIALOG.TLK" ;
    "--ftlkout", Arg.String (fun s -> output_dialogf := Some(s)), "X\temit X as new DIALOGF.TLK\n\nD Options:\n" ; 

    "--transin", Arg.String (fun s -> trans_list := !trans_list @ [s]), "X\tuse translation file X (cumulative)" ;
    "--testtrans", Arg.Set test_trans, "\ttest all translations files" ;
    "--noheader", Arg.Clear d_headers, "\tdo not emit .D header comments" ;
    "--nofrom", Arg.Clear Dlg.emit_from, "\tdo not emit .D \"// from:\" comments" ;
    "--full-from", Arg.Set two_pass, "\tGenerate complete \"// from:\" comments"; 
    "--nocom", Arg.Clear Dlg.comments, "\tdo not emit ANY .D / .BAF comments" ;
    "--transitive", Arg.Set transitive, "\tFollow EXTERN links when making D files" ;
    "--text", Arg.Set Dlg.emit_text, "\temit string text with refs in comments" ; 
    "--dout", Arg.String (fun s -> dout_list := !dout_list @ [s]), "X\tname of output .D file to emit (cumulative)" ;
    "--traify", Arg.String (fun s -> traify := Some(s)), "X\tconvert .D file X to use TRAs (use with --dout)" ;
    "--traify#", Arg.Int (fun d -> traify_num := d), "X\tstart --traify .TRA file at @X" ;
    "--extract-kits", Arg.Int (fun d -> extract_kits := d), "X\textract all kits starting with kit #X"; 
    "--forceify", Arg.String (fun s -> forceify := Some(s)), "X\tconvert .D file X to use forced strrefs (use with --dout)" ;
    "--transref", Arg.Set Dlg.use_trans_ref, "\temit string reference numbers in TRA files" ;
    "--trans", Arg.Set use_trans, "\temit coupled .D and .TRA files\n\nTLK String Options:\n" ;

    "--traify-tlk", Arg.Set extract_tlk, "\temit a .TRA file for the given .TLK file (see --textout, --min, --traify#)" ;
    "--make-tlk", Arg.String (fun s -> make_tlk := s :: !make_tlk), "X\tmake a .TLK file from .TRA file X (cumulative, see --tlkout)" ; 
    "--string", Arg.Int (fun i -> ds_list := i :: !ds_list), "X\tdisplay string reference #X (cumulative)" ;
    "--strfind", Arg.String (fun s -> strfind_list := s :: !strfind_list), "X\tdisplay strings that contain X (cumulative, regexp allowed)" ;
    "--strapp", Arg.String (fun s -> strapp_list := s :: !strapp_list), "X\tappend string X to DIALOG.TLK (cumulative)\n\nBIFF Options:\n" ;

    "--textout", Arg.String (fun s -> textout := open_out s), "X\tput text output in file X"; 
    "--textapp", Arg.String (fun s -> 
        textout := open_out_gen [Open_append ; Open_wronly ; Open_creat ; Open_text ] 511 s ) ,
        "X\tappend text output to end of file X"; 
    "--list-biffs", Arg.Set list_biff, "\tenumerate all BIFF files in CHITIN.KEY" ;
    "--list-files", Arg.Set list_files, "\tenumerate all resource files in CHITIN.KEY"; 
    "--biff", Arg.String (fun s -> bc_list := (String.uppercase s) :: !bc_list), "X\tenumerate contents of BIFF file X (cumulative)" ;
    "--biff-type", Arg.String (fun s -> bs_type_list := s :: !bs_type_list), "X\texamine all BIFF resources of extension X ... (cumulative)" ;
    "--biff-str", Arg.String (fun s -> bs_str_list := s :: !bs_str_list), "X\t... and list those containing X (cumulative, regexp allowed)" ;
    "--biff-name", Arg.Int (fun i -> Load.content_name_offset := Some(i)),
      "X\tassume matching items have a strref name at offset X" ;
    "--biff-value", Arg.Int (fun i -> biff_short := i), "X\t... or list those containing value X ..." ;
    "--biff-value-at", Arg.Int (fun i -> biff_short_at := i), "X\t... at offset X" ;
    "--biff-get", Arg.String (fun s -> bg_list := s :: !bg_list), "X\textract resource X from game BIFFs (cumulative, regexp allowed)" ;
    "--biff-get-rest", Arg.Rest (fun s -> bg_list := s :: !bg_list), "X, Y, ...\textract resources X, Y, ... from game BIFFs (regexp allowed)" ;
    "--make-biff", Arg.String (fun s -> make_biff := Some(s)), "X\tmake data\\X.bif from all files in folder X, update CHITIN.KEY" ;
    "--remove-biff", Arg.String (fun s -> remove_biff := Some(s)), "X\tremove references to biff X and its resources, update CHITIN.KEY" ;

(* 
    "--itemlist", Arg.Set make_an_itemlist, "\tmake an item listing (IWD2)" ; 
    "--xplist", Arg.Set make_an_xplist, "\tmake an XP listing (IWD2)" ; 
    "--cre_analysis", Arg.Set do_cre_analysis, "\tdo CRE analysis (IWD2)" ;
    "--itemsort", Arg.Set do_itmsort, "\tdo ITM sort (IWD2)" ;
     *)

    "", Arg.Unit (fun a -> a),  "\nARE/ITM/SPL/CRE Options:\n" ;
    "--automate", Arg.String (fun s -> automate_list := s ::
    !automate_list), "X\tautomatically make a TP2 file for ARE/ITM/SPL/CRE files in X" ;
    "--automate-min", Arg.Int (fun i -> automate_min := i),
      "X\tminimum strref # for --automate (default is SoA)";

    "--list-eff", Arg.String (fun s -> list_eff_list := s :: !list_eff_list), "X\tlist effects in resource X" ;

    "", Arg.Unit (fun a -> a),  "\nComparison Options:\n" ;
    "--cmp-from", Arg.String (fun s -> cmp_src := Some(s)), "X\temit WRITE_BYTEs to turn this file ..." ;
    "--cmp-to", Arg.String (fun s -> cmp_dest := Some(s)), "X\t... into this one";
    "--dcmp-from", Arg.String (fun s -> dcmp_src := Some(s)), "X\temit REPLACEs to turn this DLG file ..." ;
    "--dcmp-to", Arg.String (fun s -> dcmp_dest := Some(s)), "X\t... into this one";
    
    "--tcmp-from", Arg.String (fun s -> tcmp_src := Some(s)), "X\tcompare this TRA file (or directory of TRA files)..." ;
    "--tcmp-to", Arg.String (fun s -> tcmp_dest := Some(s)), "X\t... with this one (or this directory)";
    "--bcmp-from", Arg.String (fun s -> bcmp_src := Some(s)), "X\temit APPLY_BCS_PATCH to turn this BCS file..." ;
    "--bcmp-to", Arg.String (fun s -> bcmp_dest := Some(s)), "X\t... into this one" ;
    (* For debugging patch/diff: *)
    "--bcmp-orig", Arg.String (fun s -> bcmp_orig := Some(s)), "X\toriginal file to apply ..." ;
    "--bcmp-patch", Arg.String (fun s -> bcmp_patch := Some(s)), "X\t... this patch to" ;
    "--tlkcmp-from", Arg.String (fun s -> tlkcmp_src := Some(s)), "X\temit STRING_SETs to convert this TLK file ..." ;
    "--tlkcmp-to", Arg.String (fun s -> tlkcmp_dest := Some(s)), "X\t... into this one";
    "--min", Arg.Int (fun i -> user_min := Some(i)), "X\tlower range for some commands (like --tlkcmp)" ;
    "--max", Arg.Int (fun i -> user_max := Some(i)), "X\tupper range for some commands (like --string)" ;

    "", Arg.Unit (fun a -> a),  "\nLog Options:\n" ;

    "--log", Arg.String (fun s -> init_log s),"X\tlog output and details to X" ;
    "--autolog", Arg.Unit (fun () -> init_log "WSETUP.DEBUG"), "\tlog output and details to WSETUP.DEBUG" ;
    "--debug-assign", Arg.Set Var.debug_assign,"\tPrint out all values assigned to TP2 variables" ;
    "--debug-value", Arg.Set Tp.debug_pe,"\tPrint out all value expressions" ;
    "--continue", Arg.Set Tp.continue_on_error,"\tcontinue despite TP2 action errors" ;
    
    "", Arg.Unit (fun a -> a),  "\nHelp Options:\n"; 

  ] in 
  let give_help () = 
    Arg.usage argDescr usageMsg ;
    exit 1
  in 
  let handleArg str = begin 
    let base,ext = split (String.uppercase str) in 
    match ext with
    | "D" -> d_list := str :: !d_list 
    | "DLG" -> dlg_list := (base,ext) :: !dlg_list 
    | "TLK" -> Load.set_dialog_tlk_path str 
    | "TP" 
    | "TP2" -> tp_list := !tp_list @ [str] 
    | "TRA" -> trans_list := !trans_list @ [str] 
    | "ITM"
    | "EFF" 
    | "SPL" -> list_eff_list := !list_eff_list @ [str] 
    | "BCS" | "BS" -> bcs_list := !bcs_list @ [str] 
    | "BAF" -> baf_list := !baf_list @ [str] 
    | _ -> log_and_print "Unknown argument: [%s]\n\n" str ; give_help () 
  end in

  log_and_print "[%s] WeiDU version %s\n" Sys.argv.(0) version ; 

  (* see if AUTOUPDATE is in our base name *)
  begin try
    (* let update_regexp = Str.regexp_case_fold "weiduautoupdate" in  *)
    let target = Unix.getenv "weiduautoupdate" in 
    log_and_print "Auto-Updating on behalf of [%s]\n" target ; 
    let this_buff = load_file Sys.argv.(0) in 
    (* in this case we can always just copy ourselves over the target *) 
        let unlink_worked = (try Unix.unlink target ; true with _ -> false) in
        (* log_and_print "%b\n" unlink_worked ; *)
        log_and_print "\tCopying [%s] -> [%s]: " Sys.argv.(0) target ; 
        let copy_worked = try 
          let out = open_for_writing target true in
          output_string out this_buff ;
          close_out out ; 
          true
        with _ -> false in
        log_and_print "%b\n" copy_worked ;
    log_and_print "\nAuto-Updating on behalf of [%s] (done)\n" target ; 

    log_and_print
    "\n\n\t***********************************************************\n\tWeiDU has finished auto-updating all copies of itself\n\tin this directory. Please RE-RUN %s\n\tto actually install the mod.\n\t(sorry, I can't do it for you, Windows won't let me)\n" target ; 
    (if Arg.term <> "xterm" then (try ignore (read_line () ) with _ -> ()))
    ;
(*
    Sys.argv.(0) <- real_target ; 
    let env = (Unix.environment ()) in
    let fixed_env = Array.map (fun s -> 
        try let _ = Str.search_forward update_regexp s in
            "foo=bar"
        with _ -> s
        ) env in 
    let _ = Unix.execve real_target Sys.argv fixed_env in 
    *)
    exit 1 ;
  with _ -> () end ; 

  Load.game_paths := Load.registry_game_paths () ; 

  Arg.parse argDescr handleArg usageMsg  ;

  if (!auto_update_all) then begin
    (if (Arch.do_auto_update) then 
      verify_latest true);
    exit 0 ; 
  end ;

  (* see if SETUP is in our base name *)
  let setup_regexp = Str.regexp_case_fold "setup" in
  begin 
  try 
    let _ = Str.search_forward setup_regexp argv0_base 0 in
    auto () ;
  with _ -> 
    if Array.length Sys.argv <= 1 then begin 
      Arg.usage argDescr usageMsg ;
      flush_all () ; 
      log_and_print "\nEnter arguments: " ;
      let mystr = read_line () in
      if mystr = "" then exit 1
      else exit ( Sys.command (Sys.executable_name ^ " " ^ mystr))
    end ;
  end ; 

  let game = 
    if !no_game then
      Load.load_null_game () 
    else 
      Load.load_game () 
  in 

  game.Load.script_style <- !forced_script_style ; 
  Dc.cur_index := Array.length game.Load.dialog ;
  Load.saved_game := Some(game) ; 


(* 
  (  if (!make_an_itemlist) then Itemlist.make_item_list game !textout );
  (  if (!make_an_xplist) then Itemlist.make_xplist game !textout );
  ( if (!do_cre_analysis) then Itemlist.cre_analysis game !textout ) ; 
  ( if (!do_itmsort) then Itemlist.itm_randomizer game !textout ) ; 
    *)

  Automate.automate game !automate_list !automate_min !textout ;

  (match !forceify,!dout_list with
    Some(file),[dout_name] -> begin
      try 
        let name,ext = split (String.uppercase file) in 
        let inchan = open_in file in 
        let lexbuf = lex_init file inchan in 
        Dlg.local_string_ht := Some([]) ; 
        begin 
          match ext with
            "D" -> ignore (Stats.time "parsing .D files" 
                  (fun () -> Dparser.d_file Dlexer.initial lexbuf) ()  )
          | "TP2" -> ignore
                   (Stats.time "parsing .TP2 files"
                  (fun () -> Dparser.tp_file Dlexer.initial lexbuf) () )
          | _ -> log_and_print "ERROR: don't know how to --forceify files with extension [%s]\n" ext ; failwith ext 
        end ;
        pop_context (); 
        log_or_print "[%s] parsed\n" file ; 
        close_in inchan ; 

        let buf = ref (load_file file) in 

        let dout = open_for_writing dout_name true in 

        let replace lse str = 
          let my_regexp = Str.regexp (Str.quote str) in
          try 
          let num = Tlk.find_string_fast lse game.Load.dialog 
              game.Load.dialogf game.Load.dialog_search 
          in 
          let replace_with = Printf.sprintf "!%d %s" num str in 
          buf := Str.global_replace my_regexp replace_with !buf ;
          with Not_found -> 
            log_and_print "WARNING: cannot find [%s] in dialog.tlk: not --forceifying that string\n" lse.lse_male 
        in 

        (match !Dlg.local_string_ht with 
          Some(lst) -> 
          List.iter (fun  ls  -> match ls with
          | Dlg.Local_String(lse) -> 
              if lse.lse_male <> "" then begin 
                replace lse ("~" ^ lse.lse_male ^ "~" ); 
                replace lse ("%" ^ lse.lse_male ^ "%" ); 
                replace lse ("\"" ^ lse.lse_male ^ "\"" ); 
              end 
          | _ -> failwith "forceify1" 
        ) (List.rev lst); 
        | None -> failwith "forceify2" 
        ); 

        Dlg.local_string_ht := None ;

        Printf.fprintf dout "%s" !buf ; 

        close_out dout ; 
        () 

      with e -> 
        log_and_print "ERROR: problem force-ifying file [%s]: %s\n" file
          (Printexc.to_string e) ; 
        raise e
      end
  | _ -> () 
  ) ; 

  if !make_tlk <> [] then begin
    let results : (int * local_string_entry) list 
      = List.fold_left (fun acc filename ->
      let result = parse_file filename "parsing .tra files" 
        (Dparser.tra_file Dlexer.initial) in 
      log_or_print "[%s] has %d translation strings\n" filename (List.length result); 
      let result = List.rev_map (fun (i,ts) -> match ts with
        Dlg.Local_String(lse) -> (i,lse)
      | _ -> failwith "make_tlk" 
      ) result in 
      List.rev_append acc result) [] !make_tlk 
    in
    let max = 1 + (List.fold_left (fun acc (i,elt) -> 
      if i > acc then i else acc) 0 results) in
    log_and_print "New TLK will have %d entries\n" max ;
    let new_tlk = Array.make max ( { Tlk.flags = 7 ;
                                 Tlk.sound_name = "";
                                 Tlk.volume = 0; 
                                 Tlk.pitch = 0;
                                 Tlk.text = ""; } ) in 
    List.iter (fun (i,lse) ->
      let male, female = Tlk.lse_to_tlk_string lse in
      new_tlk.(i) <- male
    ) results ;

    game.Load.dialog_mod <- true ;
    game.Load.dialog <- new_tlk ; 
  end ; 


  (if !extract_tlk then begin
      let tlk = game.Load.dialog in 
      let ftlk = game.Load.dialogf in 
      let my_min = match !user_min with
        Some(i) -> i
      | None -> 0 
      in
      let my_max = match !user_max with
        Some(i) -> i
      | None -> (Array.length tlk) - 1
      in 
      let reg_list = List.map Str.regexp_case_fold !strfind_list in 
      strfind_list := [] ; 
      for i = my_min to my_max do
        let matches = reg_list = [] || 
          List.fold_left (fun acc r -> acc ||
            try 
              let _ = Str.search_forward r tlk.(i).Tlk.text 0 in
              true
            with _ -> false
          ) false reg_list 
        in 
        let escape s = 
          if not (String.contains s '~') then
            "~" ^ s ^ "~" 
          else if not (String.contains s '"') then
            "\"" ^ s ^ "\"" 
          else if not (String.contains s '%') then
            "%" ^ s ^ "%"
          else 
            failwith ("cannot --traify string [" ^ s ^ "]")
        in 
        if matches then begin 
          Printf.fprintf !textout "@%-5d =" (i + !traify_num);
          let display ts = begin
            Printf.fprintf !textout " %s" (escape ts.Tlk.text) ;
            if ts.Tlk.sound_name <> "" then 
             Printf.fprintf !textout " [%s]" ts.Tlk.sound_name ;
          end  in
          display tlk.(i) ;
          (match ftlk with 
            None -> ()
          | Some(a) -> 
            if a.(i).Tlk.text <> tlk.(i).Tlk.text ||
               a.(i).Tlk.sound_name <> tlk.(i).Tlk.sound_name then 
                display a.(i) )  ;
          Printf.fprintf !textout "\n" ; 
        end 
      done 
  end) ;

  (if !extract_kits > 0 then Kit.extract game !textout !output_dir
  !extract_kits) ;

  (match !cmp_src,!cmp_dest with
    Some(s),Some(d) ->
      let b1 = load_file s in
      let b2 = load_file d in
      let l1 = String.length b1 in
      let l2 = String.length b2 in 
      if (l1 <> l2) then begin 
        log_and_print "[%s] is %d bytes while [%s] is %d bytes\n"
          s l1 d l2 
      end else begin
        Printf.fprintf !textout "\t// patches to turn [%s] into [%s]\n" s d ;
        for i = 0 to l1 - 1 do 
          if b1.[i] <> b2.[i] then
            Printf.fprintf !textout "\tWRITE_BYTE %d %d // 0x%x == '%c'\n" 
              i (Char.code b2.[i]) (Char.code b2.[i]) b2.[i]
        done 
      end
  | _, _ -> ()) ; 


  (match !dcmp_src,!dcmp_dest with
    Some(s),Some(d) ->
        let b,e = split s in
        let buff, final_path = Load.load_resource "DLG compare command" game true b e in
        let imp_base = Filename.basename b in
        let s_dlg = Dlg.load_dlg imp_base buff in

        let b,e = split d in
        let buff, final_path = Load.load_resource "DLG compare command" game true b e in
        let imp_base = Filename.basename b in
        let d_dlg = Dlg.load_dlg imp_base buff in

        let new_buffer = Buffer.create (1024 * 32) in 
        Dlg.dlg_compare new_buffer s_dlg d_dlg game.Load.dialog game.Load.dialogf reprint_d_action ;
        Buffer.output_buffer !textout new_buffer 

  | _,_ -> ()) ; 

  (match !bcmp_src,!bcmp_dest with
    Some(s),Some(d) ->
      let b,e = split s in
      let src_buff, final_path = Load.load_resource "BCS patch command" game true b e in
      let out_name = d ^ ".patch" in
      let b,e = split d in
      let dest_buff, final_path = Load.load_resource "BCS patch command" game true b e in begin
        try begin
          let _ = Diff.create_patch src_buff dest_buff out_name 20 in
          Printf.fprintf !textout "// TP2 patch to turn %s into %s.  For example using:\n" s d;
          Printf.fprintf !textout "COPY_EXISTING ~%s~ ~override/%s~\n" s s;
          Printf.fprintf !textout "\tAPPLY_BCS_PATCH ~%s~\n" out_name
        end
        with e -> 
          Printf.printf "Failed to create patch for [%s] to [%s] : %s\n" s d
            (Printexc.to_string e)
      end
  | _,_ -> ()) ; 

(* For debugging patch/diff: *)
  (match !bcmp_orig,!bcmp_patch with
    Some(s),Some(d) ->
      let b,e = split s in
      let orig_buff, final_path = Load.load_resource "BCS patch compare command" game true b e in
      let b,e = split d in
      let patch_buff, final_path = Load.load_resource "BCS patch compare command" game true b e in begin
        try begin
          let new_buff, bad_chunks, app_chunks = Diff.do_patch orig_buff patch_buff true in begin
            if ( bad_chunks > 0 ) then begin
              log_and_print "ERROR: Cannot apply patch %s (%d bad chunks).\n" d bad_chunks ;
              failwith "Cannot Apply Patch"
            end ;
            if ( app_chunks > 0 ) then begin
              log_and_print "WARNING: %d chunks in patch file %s already applied.\n" app_chunks d
            end ;
            if (bad_chunks == 0) then
              if (new_buff = orig_buff) then
                log_and_print "File %s unchanged by patch %s.\n" s d
              else
                let out_name = s ^ ".new" in
                let out = open_out_bin out_name in begin
                  log_and_print "Saving new file to %s\n" out_name ;
                  output_string out new_buff ;
                  close_out out
                end
          end
        end
        with e -> 
          Printf.printf "Failed to patch file [%s] with patch [%s] : %s\n" s d
            (Printexc.to_string e)
      end
  | _,_ -> ()) ; 

  (match !tlkcmp_src,!tlkcmp_dest with
    Some(s),Some(d) -> 
      let stlk = Tlk.load_tlk s in
      let dtlk = Tlk.load_tlk d in
      if Array.length stlk <> Array.length dtlk then begin
        log_and_print "WARNING: %s has %d entries, %s has %d entries\n"
          s (Array.length stlk) d (Array.length dtlk) 
      end ; 
      let my_min = match !user_min with
        Some(i) -> i
      | None -> 0 
      in
      let my_max = match !user_max with
        Some(i) -> i
      | None -> (min (Array.length stlk) (Array.length dtlk)) - 1
      in 
      Printf.fprintf !textout "// TP2 File\n" ;
      for i = my_min to my_max do 
        if stlk.(i).Tlk.text <> dtlk.(i).Tlk.text then
          Printf.fprintf !textout "\tSTRING_SET %d @%d\n" i (1000000 + i) 
      done ;
      flush !textout ; 
      Printf.fprintf !textout "\n// TRA File\n" ;
      for i = my_min to my_max do 
        if stlk.(i).Tlk.text <> dtlk.(i).Tlk.text then
          Printf.fprintf !textout "@%d = ~%s~ [%s]\n"
            (1000000 + i) dtlk.(i).Tlk.text dtlk.(i).Tlk.sound_name 
      done
  | _,_ -> ()
  ) ; 

  (match !tcmp_src,!tcmp_dest with
    Some(s),Some(d) ->
      let tracompare s d = 
        try 
        let schan = open_in s in
        let lexbuf = lex_init s schan in 
        let sresult = Stats.time "parsing .TRA files" (fun () -> Dparser.tra_file Dlexer.initial lexbuf) () in
        log_or_print "[%s] parsed (%d translation strings)\n" s (List.length sresult); 
        close_in schan ; 
        pop_context (); 

        let dchan = open_in d in
        let lexbuf = lex_init d dchan in 
        let dresult = Stats.time "parsing .TRA files" (fun () -> Dparser.tra_file Dlexer.initial lexbuf) () in
        log_or_print "[%s] parsed (%d translation strings)\n" d (List.length dresult); 
        close_in dchan ; 
        pop_context (); 

        (* int * dlg.string list *)
        let left_out = ref [] in 
        List.iter (fun (si,_) ->
          let found = List.fold_left (fun acc (di,_) -> acc || si = di)
            false dresult in
          if not found then
            left_out := si :: !left_out 
        ) sresult ; 
        let left_out = List.sort compare !left_out in 
        if left_out = [] then begin
          Printf.fprintf !textout "\nAll Strings in [%s] are also in [%s]\n" s d ;
        end else begin 
          Printf.fprintf !textout 
            "\nStrings in [%s] that are not in [%s]:\n" s d ;
          List.iter (fun i ->
            Printf.fprintf !textout " %7d" i
          ) left_out ; 
          Printf.fprintf !textout "\n\n"
        end 
        with e -> 
          begin 
          Printf.fprintf !textout "\nThe ENTIRE FILE [%s] is missing:\n\t%s\n" 
            d (Printexc.to_string e) ; 
          Printf.printf "Skipping [%s] and [%s] : %s\n" s d
            (Printexc.to_string e)
          end 
      in 
      if (Unix.stat s).Unix.st_kind <> Unix.S_REG then begin
        let s_d_h = Unix.opendir s in
        try
          while true do 
            let s' = Unix.readdir s_d_h in
            let one = (s ^ "/" ^ s') in 
            let two = (d ^ "/" ^ s') in 
            if (Unix.stat one).Unix.st_kind = Unix.S_REG then 
              tracompare one two 
          done 
        with e -> (Unix.closedir s_d_h ) 
      end else begin
        tracompare s d 
      end

  | _,_ -> ()) ; 

  (* Display Strings *)
  let display_string i = 
    let male = Tlk.pretty_print game.Load.dialog i in
    let female = Tlk.pretty_print_opt game.Load.dialogf i in
    if (female = "" || male = female) then
      Printf.printf "String #%d is %s\n" i male 
    else
      Printf.printf "String #%d is %s (MALE)\nString #%d is %s (FEMALE)\n" i male i female 
  in 
  if !ds_list <> [] && ( !user_min <> None || !user_max <> None) then begin
    let my_min = match !user_min with
      Some(i) -> i
    | None -> 0 
    in
    let my_max = match !user_max with
      Some(i) -> i
    | None -> (Array.length game.Load.dialog) - 1
    in 
    for i = my_min to my_max do 
      display_string i 
    done 
  end else List.iter display_string !ds_list ;

  (* display strings that match *)
  if (!strfind_list <> []) then begin
    let reg_list = List.map Str.regexp_case_fold !strfind_list in 
    Array.iteri (fun i s ->
      let matches_one = 
        List.fold_left (fun acc r -> acc ||
          try 
            let _ = Str.search_forward r s.Tlk.text 0 in
            true
          with _ -> false
        ) false reg_list 
      in
      if matches_one then 
        Printf.printf "String #%d is %s\n" i (Tlk.pretty_print game.Load.dialog i)
    ) game.Load.dialog 
  end ; 

  (* List all BIFFs *)
  if (!list_biff) then begin
    Key.list_biff game.Load.key !textout 
  end ;

  (* List all files *)
  if (!list_files) then begin
    Key.list_key game.Load.key !textout 
  end ;

  (* List BIFF contents *)
  if (!bc_list <> []) then begin
    Key.list_biff_contents game.Load.key !textout !bc_list
  end ; 

  (* Regex on BIFF contents *)
  if (!bs_type_list <> [] && !bs_str_list <> []) then begin
    Load.search_biff_contents game !textout !bs_type_list !bs_str_list
  end else if (!bs_type_list <> [] && !biff_short_at <> 0) then begin
    let size = if !biff_short > 65535 then 4
               else if !biff_short > 255 then 2
               else 1
    in 
    Load.search_biff_contents_fun game !textout !bs_type_list
      (fun buff -> 
        (String.length buff) >= (!biff_short_at + size) &&
        let i = (match size with 
        | 2 -> short_of_str_off 
        | 1 -> byte_of_str_off
        | _ -> int_of_str_off
        ) buff !biff_short_at in
        i = !biff_short)
  end else if (!bs_type_list <> [] || !bs_str_list <> []) then begin
    log_and_print "WARNING: Please specify both --biff-type EXT and (--biff-str STRING or --biff-short-at OFFSET)\n"
  end  ;

  (* Grab resources from BIFFs *) 
  if (!bg_list <> []) then begin
    let files_in_chitin = Key.list_of_key_resources game.Load.key in 

    let try_to_load str = begin
      try begin
        let base,ext = split (String.uppercase str) in 
        let path = !output_dir ^ "/" ^ str in 
        let out = open_for_writing path true in
        if ext <> "IDS" && ext <> "2DA" then begin
          let fullpath : string = Load.copy_resource game base ext out in
          close_out out ;
          log_and_print "[%s] created from [%s]\n" path fullpath 
        end else begin
          let buff, fullpath = Load.load_resource "--biff-get" game false  base ext in
          output_string out buff ;
          close_out out ;
          log_and_print "[%s] created from [%s]\n" path fullpath 
        end
      end with e ->
        Printf.printf "[%s] --biff-get error: %s\n" str (Printexc.to_string e)
    end in 

    List.iter (fun str -> 
      try 
        let any_matches = ref false in 
        let regexp = Str.regexp_case_fold str in 
        List.iter (fun possible ->
          if Str.string_match regexp possible 0 then begin
            any_matches := true ;
            try_to_load possible 
          end 
        ) files_in_chitin ;
        if not !any_matches then
          try_to_load str 
      with e ->
        log_and_print "\nERROR: %s\n" (Printexc.to_string e)  
    ) !bg_list 
  end ;

  (match !make_biff with
  | None -> () 
  | Some(s) -> begin
    let file_list = ref [] in 
    let s_d_h = Unix.opendir s in
    (try
      while true do 
        let s' = Unix.readdir s_d_h in
        file_list := (s ^ "/" ^ s') :: !file_list 
      done
    with _ -> () ) ;
    if !file_list <> [] then begin 
      let filename = "data/" ^ s ^ ".bif" in 
      let new_key = Biff.save_biff game.Load.key filename !file_list in
      let oc = open_for_writing "CHITIN.KEY" true in 
      Key.save_key new_key oc ;
      close_out oc 
    end 
    end 
  ) ; 

  (match !remove_biff with
  | None -> ()
  | Some(f) -> 
    let new_key = Key.remove_biff game.Load.key f in 
    let oc = open_for_writing "CHITIN.KEY" true in 
    Key.save_key new_key oc ;
    close_out oc 
  ) ; 


  (* Append Strings *)
  let lse_strapp_list = List.map (fun s -> 
    Dlg.Local_String( { lse_male = s; lse_female = s;
    lse_male_sound = "" ; lse_female_sound = ""; })) !strapp_list in
  if (lse_strapp_list <> []) then begin
    let _ = List.map (Dc.resolve_tlk_string game) lse_strapp_list in
    ()
  end ; 


  (* Handle DLG files *)
  let loaded_dlgs = List.map (fun (b,e) -> 
    try 
      let buff, final_path = Load.load_resource "DLG decompile command" game true b e in
      let imp_base = Filename.basename b in
      let dlg = 
        try Dlg.load_dlg imp_base buff 
        with e -> log_and_print "ERROR: problem loading [%s]: %s\n" b
        (Printexc.to_string e) ; raise e
      in 
      let out_name = match !dout_list with
        hd :: tl -> dout_list := tl ; 
          if (Filename.is_implicit hd) then 
            !output_dir ^ "/" ^ hd 
          else 
            hd 
      | [] -> !output_dir ^ "/" ^ imp_base ^ ".D" 
      in 
      let transout_name = (Filename.chop_extension out_name ^ ".TRA" ) in 
      (dlg,out_name,transout_name,b,e,final_path) 
    with e -> log_and_print "ERROR: problem handling [%s]: %s\n" b
        (Printexc.to_string e) ; raise e
    ) !dlg_list 
  in 
  for i = 1 to if !transitive || !two_pass then 2 else 1 do 
  List.iter (fun (dlg,out_name,transout_name,b,e,final_path) -> 
    try 
      let out_chan = open_for_writing out_name false in
      let out_trans_chan = match !use_trans with
        true -> 
          log_and_print "[%s] created as translation file\n" transout_name ;
          Some(open_for_writing transout_name false) 
      | false -> None
      in 
      if (!d_headers) && !Dlg.comments then begin
        Printf.fprintf out_chan "// creator  : %s (version %s)\n" Sys.argv.(0) version; 
        Printf.fprintf out_chan "// argument : %s.%s\n" b e ;
        Printf.fprintf out_chan "// game     : %s\n" game.Load.game_path;
        Printf.fprintf out_chan "// source   : %s\n" final_path ;
        Printf.fprintf out_chan "// dialog   : %s\n" game.Load.dialog_path ;
        Printf.fprintf out_chan "// dialogF  : %s\n\n" game.Load.dialogf_path ;
      end ; 
      let new_buff = Buffer.create (1024 * 32) in 
      Dlg.emit_d dlg out_name game.Load.dialog game.Load.dialogf new_buff out_trans_chan None reprint_d_action !transitive ;
      Buffer.output_buffer out_chan new_buff ;
      close_out out_chan ; 
    with e -> 
      log_and_print "ERROR: problem creating [%s] from [%s]: %s\n" out_name
        b (Printexc.to_string e) ; 
      raise e
  ) loaded_dlgs ; 
  done ;

  (* Handle TRA files *)
  Dc.ok_to_resolve_strings_while_loading := Some(game) ; 
  List.iter handle_tra_filename !trans_list ; 

  if !test_trans then begin
    Dc.test_trans !textout game
  end ; 

  (match !traify,!dout_list with
    Some(file),[dout_name] -> begin
      try 
        let name,ext = split (String.uppercase file) in 
        let inchan = open_in_bin file in 
        let lexbuf = lex_init file inchan in 
        Dlg.local_string_ht := Some([]) ; 

        let old_ok = !Dc.ok_to_resolve_strings_while_loading in
        Dc.ok_to_resolve_strings_while_loading := None ; 
        Dc.doing_traify := true ; 

        begin 
          match ext with
            "D" -> ignore (Stats.time "parsing .D files" 
                  (fun () -> Dparser.d_file Dlexer.initial lexbuf) ()  )
          | "TP2" -> ignore
                   (Stats.time "parsing .TP2 files"
                  (fun () -> Dparser.tp_file Dlexer.initial lexbuf) () ) 
          | "BAF" -> 
            (if (!Dlg.local_string_ht = None) then
              Dlg.local_string_ht := Some([]) ) ;
            ignore (Stats.time "parsing .BAF files"
              (fun () -> Bafparser.baf_file Baflexer.initial lexbuf) () )
          | _ -> log_and_print "ERROR: don't know how to --traify files with extension [%s]\n" ext ; failwith ext 
        end ;
        log_or_print "[%s] parsed for --traify\n" file ; flush stdout; 
        pop_context (); 
        close_in inchan ; 

        Dc.ok_to_resolve_strings_while_loading := old_ok ; 
        Dc.doing_traify := false; 

        let buf = ref (load_file file) in 

        let base = 
          try (Filename.chop_extension dout_name)
          with _ -> dout_name 
        in 

        let transout_name = base ^ ".TRA" in 
        let dout_name = base ^ "." ^ ext in 

        let dout = open_for_writing dout_name true in 

        let traout = open_for_writing transout_name true in 

        let counter = traify_num in 

        (* replace the given string with the tra-value of the counter *) 
        let replace str = 
          let my_regexp = Str.regexp (Str.quote str) in
          let replace_with = Printf.sprintf "@%d" !counter in 
          buf := Str.global_replace my_regexp replace_with !buf ;
        in 
        let remove str = 
          let my_regexp = Str.regexp (Str.quote str) in
          buf := Str.global_replace my_regexp "" !buf ;
        in 

        (match !Dlg.local_string_ht with 
          Some(lst) -> 
        List.iter (fun  ls  -> 
          match ls with
          | Dlg.Local_String(lse) -> 
              if lse.lse_male <> "" then begin 
                Printf.fprintf traout "@%-4d = ~%s~" !counter lse.lse_male ;
                replace ("~" ^ lse.lse_male ^ "~" ); 
                replace ("%" ^ lse.lse_male ^ "%" ); 
                replace ("\"" ^ lse.lse_male ^ "\"" ); 
                if lse.lse_male_sound <> "" then begin
                  Printf.fprintf traout " [%s]" lse.lse_male_sound ; 
                  remove ("[" ^ lse.lse_male_sound ^ "]")
                end ; 
              end ; 
              if lse.lse_female <> "" && 
                 lse.lse_female <> lse.lse_male then begin
                Printf.fprintf traout " ~%s~" lse.lse_female ; 
                remove ("~" ^ lse.lse_female ^ "~" ); 
                remove ("%" ^ lse.lse_female ^ "%" ); 
                remove ("\"" ^ lse.lse_female ^ "\"" ); 
                if lse.lse_female_sound <> "" then begin
                  Printf.fprintf traout " [%s]" lse.lse_female_sound ; 
                  remove ("[" ^ lse.lse_female_sound ^ "]")
                end 
              end ; 
              Printf.fprintf traout "\n" ; 
              incr counter ;
              () 
          | _ -> failwith "traify1" 
        ) (List.rev lst); 
        | None -> failwith "traify2" 
        ); 

        Dlg.local_string_ht := None ;

        Printf.fprintf dout "%s" !buf ; 

        close_out traout ;
        close_out dout ; 
        () 

      with e -> 
        log_and_print "ERROR: problem tra-ifying file [%s]: %s\n" file
          (Printexc.to_string e) ; 
        raise e
      end
  | _ -> () 
  ) ; 



  List.iter (fun str -> 
    try 
      let script = handle_baf_filename str in
      let name,ext = split (Filename.basename str) in 
      let out = open_out_bin (!output_dir ^ "/" ^ name ^ ".bcs") in
      Bcs.save_bcs game (Bcs.Save_BCS_OC(out)) script ;
      close_out out 
    with e -> log_and_print "ERROR: problem loading [%s]: %s\n" str
      (Printexc.to_string e) ; raise e
  ) !baf_list ; 


  (* Handle D files *)
  List.iter (handle_d_filename ) !d_list ;
  Dc.ok_to_resolve_strings_while_loading := None;

  (* Emit DLG files *) 
  emit_dlg_files game !output_dir ;

  (* Check that we can write to the given TLK file *) 
  (match !output_dialog with
    Some(path) when file_exists path -> begin
        try Unix.access path [Unix.W_OK] ;
            log_or_print "[%s] claims to be writeable.\n" path ; 
            if (Unix.stat path).Unix.st_kind <> Unix.S_REG then
              failwith (path ^ " is a not a regular file") ;
            log_or_print "[%s] claims to be a regular file.\n" path ; 
            ()
        with e -> 
          log_and_print "\nERROR: The file [%s] cannot be written to.
Perhaps it is in use by another process (close ShadowKeeper, all Infinity
Engine games and editors, etc.). It may also be naturally read-only: use
Windows Explorer and right-click on the file to pull up its properties.
Make sure that the \"read-only\" box is NOT checked. Please fix this
problem and try again.\n" path ;
          pause_at_end := true ; 
          raise e 
      end 
    | _ -> () 
  ) ; 

  if !tp_list <> [] then begin
    pause_at_end := true ; 
    if !Tp.always_uninstall then pause_at_end := false ; 
    load_log () ; 
    let q = Queue.create () in 
    List.iter (fun tp_file -> Queue.add tp_file q) !tp_list ;
    while not (Queue.is_empty q) do
      let tp_file = Queue.take q in 
      try 
      if file_exists tp_file then begin 
        let result = handle_tp2_filename tp_file in 
        Tp.handle_tp game handle_tp2_filename handle_tra_filename
          (handle_d_filename) 
          (compile_baf_filename game) 
          (handle_script_buffer) 
          (handle_dlg_buffer) 
          (handle_d_buffer) 
          (fun filename -> 
            log_only "Enqueuing [%s] for TP2 processing.\n" filename ; 
            Queue.add filename q)
          emit_dlg_files 
          tp_file result
      end
      with e -> 
        log_and_print "ERROR: problem parsing TP file [%s]: %s\n" tp_file
          (Printexc.to_string e) ; 
        raise e
      done 
  end ;

  List.iter (fun str ->
    let name,ext = split (String.uppercase str) in 
    let buff,path = Load.load_resource "list effects command" game true name ext in
    Printf.fprintf !textout "[%s] has effects:\n" str ;
    let eff_arr = match ext with
    | "EFF" -> Load.eff_of_eff buff
    | _ -> Load.eff_of_spl_itm buff 
    in
    Array.iter (fun eff ->
      let op = eff.Load.opcode in 
      let eff_name = Eff_table.name_of_opcode op in
      if op = 139 then begin (* display string *)
        Printf.fprintf !textout "\t%s %s #%d\n" eff_name 
          (Tlk.pretty_print game.Load.dialog eff.Load.arg1)
          eff.Load.arg1
      end else if op = 101 then begin
        Printf.fprintf !textout "\t%s (%s)\n" eff_name 
          (Eff_table.name_of_opcode eff.Load.arg2)
      end else begin
        Printf.fprintf !textout "\t%s\n" eff_name 
      end
    ) eff_arr
  ) !list_eff_list ;

  (* Handle BCS files *)
  List.iter (fun str -> 
    let b,e = split str in 
    try 
      let buff, _ = 
        if file_exists str then (load_file str),"" else 
        Load.load_resource "decompile BCS command" game true b (String.uppercase e) 
      in
      let script =  handle_script_buffer str buff in
      let base = Filename.basename b in 
      let out_name = !output_dir ^ "/" ^ base ^ ".BAF" in 
      let out = open_out out_name in 
      (try 
        Bcs.print_script_text game (Bcs.Save_BCS_OC(out)) 
          (Bcs.BCS_Print_Script(script)) (!Dlg.comments) None ;
        close_out out 
      with e -> 
        log_and_print "ERROR: problem printing script [%s]: %s\n" b 
          (Printexc.to_string e) ; close_out out 
      )
    with e -> log_and_print "ERROR: problem handling [%s]: %s\n" b
        (Printexc.to_string e) 
    ) !bcs_list ;

  (match !backup_list_chn with
    Some(c) -> close_out c ; backup_list_chn := None 
  | None -> () ) ;
  backup_dir := None ; 

  (* make sure we add all those strings! *)
  if not (Queue.is_empty Dc.strings_to_add) then begin
    if (!output_dialog = None && !output_dialogf = None) then begin
      log_or_print "You did not specify '--tlkout dialog.tlk', so %d strings were not saved.\n" (Queue.length Dc.strings_to_add);
    end else begin 
      let dc_lse_strapp_list = Dc.strings_to_add in 
      Load.append_strings game dc_lse_strapp_list 
    end 
  end ;

  if not (game.Load.str_sets = []) then begin
    log_or_print "WARNING: %d SET_STRINGs were executed but no uninstall information was created.\n" (List.length game.Load.str_sets)
  end ;

  List.iter (fun str ->
    let name,ext = split (String.uppercase str) in 
    let tlk = Tlk.load_tlk str in
    let max = 
      if Array.length tlk > Array.length game.Load.dialog then
        Array.length game.Load.dialog
      else
        Array.length tlk 
    in 
    for i = 0 to max - 1 do 
      game.Load.dialog.(i) <- tlk.(i)
    done ;
    game.Load.dialog_mod <- true
  ) !tlk_merge ; 

  (* Emit DIALOG.TLK *)
  (match !output_dialog, game.Load.dialog_mod with
    Some(path), true -> 
      let outchan = open_for_writing path true in 
      Tlk.save_tlk path game.Load.dialog outchan 
  | _, _ -> ()) ; 

  (* Emit DIALOGF.TLK *) 
  (match !output_dialogf, game.Load.dialogf, game.Load.dialogf_mod with
    Some(path),Some(t),true -> 
      let outchan = open_for_writing path true in
      Tlk.save_tlk path t outchan 
  | _, _, _ -> () ) ;

  List.iter (fun str -> log_and_print "%s" str) !Tp.strings_to_print_at_exit ;

  ()
;;


(try 
  main () 
with e -> 
  log_and_print "\nFATAL ERROR: %s\n" (Printexc.to_string e) ) 
  
;;

(match !Util.log_channel with
  Some(o) -> Stats.print o "\n\t\tWeiDU Timings\n" ; flush o 
| None -> () )

;; 

if !pause_at_end then begin
  Printf.printf "\nPress ENTER to exit.\n" ;
  try ignore (read_line () ) with _ -> () 
end 

;;

List.iter (fun s -> 
  log_or_print "Executing: [%s]\n" s ; 
  ignore (Unix.system (Arch.slash_to_backslash s))) 
    !execute_at_exit 

;;

(match !Util.log_channel with
  Some(o) -> close_out o 
| None -> () ) 

;;

Util.log_channel := None 

;;

exit 0 
;;
