(* Talk-Patcher / Installer *)
open Util
open Diff

let always_yes = ref false 
let always_uninstall = ref false 
let continue_on_error = ref false 

let debug_pe = ref false 

let write_empty_inv_slot buff off itm =
  let i = ref 0 in
  while !i < 16 do
  let slot = short_of_str_off buff (off + (!i * 2)) in
  if slot = 65535 then begin
  write_short buff (off + (!i * 2)) itm;
  i := 16 ;
  end else
  incr i
  done;

type tp_flag =
  Auto_Tra of string
| Menu_Style of string
| Ask_Every_Component 
| Always of tp_action list 
| Allow_Missing of string list
| Script_Style of Load.script_style 

and tp_file = {
  backup    : string ;
  author    : string ; 
  flags     : tp_flag list ; 
  languages : tp_lang list ;
  module_list : tp_mod list ;
} 

and tp_lang = {
  lang_name : string ;
  lang_dir_name : string ;
  lang_tra_files : string list ;
} 

and tp_mod = {
  mod_name : Dlg.tlk_string ;
  mod_parts : tp_action list;
  mod_flags : tp_mod_flag list ; 
} 

and tp_mod_flag = 
  TPM_Deprecated of Dlg.tlk_string (* should be uninstalled when encountered *) 
| TPM_RequireComponent of string * int * Dlg.tlk_string  
| TPM_ForbidComponent of string * int * Dlg.tlk_string  
| TPM_RequirePredicate of predicate * Dlg.tlk_string
| TPM_SubComponents of Dlg.tlk_string * predicate
| TPM_Designated of int
| TPM_NotInLog

and tp_copy_args = {
  copy_get_existing   : bool ;  (* get from biffs? *)
  copy_use_regexp     : bool ;
  copy_use_glob       : bool ;
  copy_file_list      : ( string * string ) list ; (* (source,dest) list *)
  copy_patch_list     : tp_patch list ;
  copy_constraint_list: tp_constraint list ;
  copy_backup         : bool ; (* normally TRUE *)
} 

and store_args = {
  overwrite_store_item  : bool ; (* normally TRUE *)
}

and tp_action =
  | TP_Copy   of tp_copy_args 
  | TP_CopyRandom of string list * tp_patch list 
  | TP_RandomSeed of int 
  | TP_Compile of (string list) * (string list) (* DLG, TRA *)
  | TP_Inlined_File of (string * string)
  | TP_Mkdir of string list 
  | TP_Require_File of string * (Dlg.tlk_string)
  | TP_Forbid_File of string * (Dlg.tlk_string)
  | TP_Append of string * string * (tp_constraint list)
  | TP_Append_Col of string * (string list) * (tp_constraint list)
  | TP_Set_Col of string * (string list) * int 
  | TP_Extend_Top of bool * string * string * (tp_patch list) * (string list)
  | TP_Extend_Bottom of bool * string * string * (tp_patch list) 
      * (string list) 
  | TP_At_Exit of string 
  | TP_At_Interactive_Exit of string 
  | TP_At_Uninstall of string 
  | TP_At_Interactive_Uninstall of string 
  | TP_Add_Kit of tp_add_kit
  | TP_Add_Music of tp_add_music 
  | TP_Add_Projectile of tp_add_projectile 
  | TP_String_Set of string * Dlg.tlk_string * (string option)
  | TP_Fail of Dlg.tlk_string 
  | TP_Print of Dlg.tlk_string 
  | TP_If of predicate * (tp_action list) * (tp_action list)  
  | TP_Uninstall_Now of string * int 

and predicate = 
  | Pred_And of predicate * predicate
  | Pred_Or of predicate * predicate
  | Pred_Not of predicate
  | Pred_File_MD5 of string * string
  | Pred_File_Exists of string
  | Pred_File_Exists_In_Game of string
  | Pred_File_Size of string * int
  | Pred_File_Contains of string * string
  | Pred_True
  | Pred_Expr of tp_patchexp 

and tp_add_kit = {
  kit_name : string ;
  clasweap : string ;
  weapprof : string ;
  abclasrq : string ;
  abclsmod : string ;
  abdcdsrq : string ;
  abdcscrq : string ;
  dualclas : string ;
  alignmnt : string ;
  ability_file : string ; 
  include_in : string ;
  lower : Dlg.tlk_string ;
  mixed : Dlg.tlk_string ;
  help : Dlg.tlk_string ;
  unused_class : string ;
  tob_start : string list ;
  tob_abbr : string ; 
} 

and tp_add_map_note = {
  xcoord : int ;
  ycoord : int ;
  mstring : Dlg.tlk_string ;
  colour : string ;
}

and tp_add_cre_item = {
  item_name : string ;
  i_charge1 : int ;
  i_charge2 : int ;
  i_charge3 : int ;
  i_flags : string ;
  item_slot : string ;
  equip : bool ; (* Normally FALSE *)
  twohanded_weapon : bool ; (* Normally TRUE *)
}

and tp_add_music = {
  music_name : string ;
  music_file : string ; 
} 

and tp_add_projectile = {
  pro_file : string ;
}

and tp_patch = 
  | TP_PatchStrRef of tp_patchexp * Dlg.tlk_string (* offset + text *)
  | TP_PatchString of string * Dlg.tlk_string (* regexp + text *) 
  | TP_PatchStringTextually of string * string (* regexp + text *) 
  | TP_PatchStringEvaluate of string * (tp_patch list) * string (* see below *)
  | TP_PatchReplaceBCSBlock of string * string (* old + new *) 
  | TP_PatchReplaceBCSBlockRE of string * string (* old + new *) 
  | TP_PatchApplyBCSPatch of string (* patch *) * (string option) (* copyover *)
  | TP_PatchByte of tp_patchexp * tp_patchexp
  | TP_PatchShort of tp_patchexp * tp_patchexp
  | TP_PatchLong of tp_patchexp * tp_patchexp
  | TP_PatchReadAscii of tp_patchexp * string
  | TP_PatchReadByte of tp_patchexp * string
  | TP_Add_Known_Spell of string * int * string
  | TP_PatchReadShort of tp_patchexp * string
  | TP_PatchReadLong of tp_patchexp * string
  | TP_Read2DA of tp_patchexp * tp_patchexp * tp_patchexp * string 
  | TP_PatchASCII of tp_patchexp * string * bool (* evaluate? *) 
  | TP_PatchInsertBytes of tp_patchexp * tp_patchexp 
  | TP_PatchDeleteBytes of tp_patchexp * tp_patchexp 
  | TP_PatchSet of string * tp_patchexp 
  | TP_PatchWhile of tp_patchexp * (tp_patch list)
  | TP_PatchFor of 
    (tp_patch list) * tp_patchexp * (tp_patch list) * (tp_patch list)
  | TP_PatchIf of tp_patchexp * (tp_patch list) * (tp_patch list)
  | TP_Patch2DA of tp_patchexp * tp_patchexp * tp_patchexp * tp_patchexp
  | TP_Add_Map_Note of tp_add_map_note
  | TP_Patch_Gam of string * string * int * int
  | TP_Add_Cre_Item of tp_add_cre_item
  | TP_Add_S_Item of store_args * string * int * int * int * string * int
  | TP_Remove_Known_Spell of string list
  | TP_PatchWriteFile of tp_patchexp * string * bool (* where, what, insert? *)
  | TP_CompileBAFtoBCS
  | TP_CompileBCStoBAF
  | TP_CompileDtoDLG
  | TP_CompileDLGtoD


and tp_constraint = 
  | TP_Contains of string
  | TP_NotContains of string 
  | TP_IfSizeIs of int 
  | TP_Eval of tp_patchexp 
  | TP_ButOnlyIfItChanges

and tp_patchexp =
  | PE_String of string
  | PE_StringEqual of string * string * bool (* ignore-case? *)
  | PE_Not of tp_patchexp
  | PE_Add of tp_patchexp * tp_patchexp 
  | PE_Sub of tp_patchexp * tp_patchexp 
  | PE_Mul of tp_patchexp * tp_patchexp 
  | PE_Div of tp_patchexp * tp_patchexp 
  | PE_Equal of tp_patchexp * tp_patchexp 
  | PE_And of tp_patchexp * tp_patchexp 
  | PE_Or of tp_patchexp * tp_patchexp 
  | PE_GT of tp_patchexp * tp_patchexp 
  | PE_GTE of tp_patchexp * tp_patchexp 
  | PE_LT of tp_patchexp * tp_patchexp 
  | PE_LTE of tp_patchexp * tp_patchexp 

  | PE_BAND of tp_patchexp * tp_patchexp 
  | PE_BOR of tp_patchexp * tp_patchexp 
  | PE_BNOT of tp_patchexp 
  | PE_BXOR of tp_patchexp * tp_patchexp 
  | PE_BLSL of tp_patchexp * tp_patchexp 
  | PE_BLSR of tp_patchexp * tp_patchexp 
  | PE_BASR of tp_patchexp * tp_patchexp 

  | PE_Random of tp_patchexp * tp_patchexp 

  | PE_If of tp_patchexp * tp_patchexp * tp_patchexp 

let strings_to_print_at_exit : string list ref = ref [] 

let rec get_menu_style fl = match fl with
    [] -> 0
  | Menu_Style(i) :: tl -> int_of_string i
  | hd :: tl -> get_menu_style tl 

let rec pe_to_str pe = match pe with
  | PE_String(s) -> s
  | PE_StringEqual(s1,s2,b) -> Printf.sprintf "%s %s %s" 
      s1 (if b then "STRING_COMPARE_CASE" else "STRING_COMPARE") s2 
  | PE_Not(e) -> Printf.sprintf "NOT (%s)" (pe_to_str e) 
  | PE_Add(e1,e2) -> Printf.sprintf "(%s) + (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_Sub(e1,e2) -> Printf.sprintf "(%s) - (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_Mul(e1,e2) -> Printf.sprintf "(%s) * (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_Div(e1,e2) -> Printf.sprintf "(%s) / (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_Equal(e1,e2) -> Printf.sprintf "(%s) = (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_And(e1,e2) -> Printf.sprintf "(%s) AND (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_Or(e1,e2) -> Printf.sprintf "(%s) OR (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_GT(e1,e2) -> Printf.sprintf "(%s) > (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_GTE(e1,e2) -> Printf.sprintf "(%s) >= (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_LT(e1,e2) -> Printf.sprintf "(%s) < (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_LTE(e1,e2) -> Printf.sprintf "(%s) <= (%s)" (pe_to_str e1) (pe_to_str e2)

  | PE_BAND(e1,e2) -> Printf.sprintf "(%s) BAND (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_BOR(e1,e2) -> Printf.sprintf "(%s) BOR (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_BXOR(e1,e2) -> Printf.sprintf "(%s) BXOR (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_BLSL(e1,e2) -> Printf.sprintf "(%s) BLSL (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_BLSR(e1,e2) -> Printf.sprintf "(%s) BLSR (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_BASR(e1,e2) -> Printf.sprintf "(%s) BASR (%s)" (pe_to_str e1) (pe_to_str e2)
  | PE_BNOT(e1) -> Printf.sprintf "BNOT (%s)" (pe_to_str e1)

  | PE_Random(e1,e2) -> Printf.sprintf "RANDOM((%s) (%s))" (pe_to_str e1) (pe_to_str e2)

  | PE_If(e1,e2,e3) -> Printf.sprintf "(%s) ? (%s) : (%s)" 
    (pe_to_str e1) (pe_to_str e2) (pe_to_str e3) 

(************************************************************************
 * For handling lists of modules.
 ************************************************************************)
let get_nth_module tp_file n =
  let lst = tp_file.module_list in 
  let last = ref (-1) in
  let rec process lst = match lst with
  | [] -> 
    log_or_print "%s's %dth component not found." tp_file.backup n ; 
    raise Not_found
  | hd :: tl -> 
    begin
      (match (List.find_all (fun x -> match x with
      | TPM_Designated(i) -> last := i ; true
      | _ -> false) hd.mod_flags) with
      | [] -> incr last
      | _ -> ()) ;
      if (!last = n) then hd else process tl 
    end 
  in
  process lst 

let get_last_module_index tp_file = 
  let last = ref 0 in
  List.fold_left (fun acc elt -> 
    let this_one = 
      let rec process lst = match lst with
      | TPM_Designated(i) :: tl -> i
      | hd :: tl -> process tl 
      | [] -> !last + 1 
      in process elt.mod_flags
    in 
    last := this_one ;
    max !last acc
  ) 0 tp_file.module_list 


(************************************************************************
 * Set some default strings that can be overwritten. 
 ************************************************************************)
let init_default_strings () = 
  let add i s = 
    Dc.add_trans_strings [i,(Dlg.Local_String({
      lse_male = s; 
      lse_male_sound = s;
      lse_female = s;
      lse_female_sound = s;
    }))]
  in 
  add (-1000) "This mod has" ;
  add (-1001) "distinct optional components.\nTo save time, you can choose what to do with them at a high level rather\nthan being asked about each one.\n" ;
  add (-1002) "What should be done with all components that are NOT YET installed?\n[I]nstall them, [S]kip them, [A]sk about each one? " ;
  add (-1003) "What should be done with all components that are ALREADY installed?\n[R]e-install them, [U]ninstall them, [S]kip them, [A]sk about each one? " ;

  add (-1004) "PLEASE email the file" ;
  add (-1005) "to" ;

  add (-1006) "Install Component [" ;
  add (-1007) "]\n[R]e-Install, [N]o Change or [U]ninstall or [Q]uit? " ;
  add (-1008) "]\n[Y]es or [N]o or [Q]uit? " ;

  add (-1009) "Enter part of a module name: " ; 

  add (-1010) "Re-Installing [";
  add (-1011) "] component " ; 
  add (-1012) "Re-Installing Using Language" ;

  add (-1013) "Removing old installation of [" ;
  add (-1014) "] first ..." ;
  add (-1015) "SUCCESSFULLY REMOVED OLD" ;

  add (-1016) "Installing" ;
  add (-1017) "ERROR Installing [" ;
  add (-1018) "], rolling back to previous state" ;
  add (-1019) "SUCCESSFULLY INSTALLED" ;
  add (-1020) "Skipping" ;
  add (-1021) "Removing [";
  add (-1022) "] (component #";
  add (-1023) ")" ;
  add (-1024) "SUCCESSFULLY REMOVED [" ;

  add (-1025) "]\n[R]e-install, [N]o Change, [U]ninstall, [Q]uit or choose one:" ;
  add (-1026) "]\n[N]o, [Q]uit or choose one:" ;
  add (-1027) " (currently installed)";
  ()


(************************************************************************
 * Evaluate a TP2 Patch Expression
 ************************************************************************)
let is_true i = (Int32.compare i (Int32.zero)) <> 0 

let eval_pe_warn = ref true  

let rec eval_pe p = 
  match p with
  | PE_String(s) -> 
    begin
      try
        Int32.of_string (Var.substitute ("%" ^ s ^ "%"))
      with e -> 
        begin
          try
            Int32.of_string (Var.substitute (s))
          with e -> 
            begin
              (if !eval_pe_warn then log_and_print 
                "ERROR: cannot convert %s or %%%s%% to an integer  (yields: %s %s)\n" s s (Var.substitute ("%" ^ s ^ "%")) (Var.substitute s)) ;
              raise e
            end 
        end 
    end 

(*
      (try Int32.of_string (Var.substitute s)
       with e -> 
       (if (!eval_pe_warn) then log_and_print "ERROR: cannot convert ~%s~ (or ~%s~) to an integer\n" s (Var.substitute s)) ; raise e)
       *)
  | PE_Random(lb,ub) -> 
    let lb = eval_pe lb in
    let ub = eval_pe ub in
    if lb > ub then 0l 
    else if lb = ub then lb
    (* random(3,5) = 3 + rand_zero_exclusive((5-3)+1) *)
    else Int32.add lb (Random.int32 (Int32.succ (Int32.sub ub lb)))

  | PE_StringEqual(s1,s2,ignore_case) ->
      let s1 = Var.substitute s1 in 
      let s2 = Var.substitute s2 in 
      let comparison = if ignore_case then 
        (fun s1 s2 -> String.compare (String.uppercase s1)
        (String.uppercase s2)) else String.compare 
      in 
      Int32.of_int (comparison s1 s2) 
  | PE_Add(a,b) -> let a = eval_pe a in let b = eval_pe b in Int32.add a b 
  | PE_Sub(a,b) -> let a = eval_pe a in let b = eval_pe b in Int32.sub a b 
  | PE_Mul(a,b) -> let a = eval_pe a in let b = eval_pe b in Int32.mul a b 
  | PE_Div(a,b) -> let a = eval_pe a in let b = eval_pe b in 
    (try Int32.div a b with Division_by_zero -> Int32.zero)
  | PE_Not(a) ->   if is_true (eval_pe a) 
                   then Int32.zero else Int32.one
  | PE_And(a,b) -> if is_true (eval_pe a) && is_true (eval_pe b) 
                  then Int32.one else Int32.zero 
  | PE_Or(a,b) -> if is_true (eval_pe a) || is_true (eval_pe b) 
                  then Int32.one else Int32.zero 
  | PE_Equal(a,b) -> if Int32.compare (eval_pe a) (eval_pe b) = 0 
                  then Int32.one else Int32.zero 
  | PE_GT(a,b) -> if Int32.compare (eval_pe a) (eval_pe b) > 0 
                  then Int32.one else Int32.zero 
  | PE_GTE(a,b) -> if Int32.compare (eval_pe a) (eval_pe b) >= 0
                  then Int32.one else Int32.zero 
  | PE_LT(a,b) -> if Int32.compare (eval_pe a) (eval_pe b) < 0 
                  then Int32.one else Int32.zero 
  | PE_LTE(a,b) -> if Int32.compare (eval_pe a) (eval_pe b) <= 0 
                  then Int32.one else Int32.zero 

  | PE_BAND(a,b) -> Int32.logand (eval_pe a) (eval_pe b) 
  | PE_BOR(a,b) -> Int32.logor (eval_pe a) (eval_pe b) 
  | PE_BXOR(a,b) -> Int32.logxor (eval_pe a) (eval_pe b) 
  | PE_BNOT(a) -> Int32.lognot (eval_pe a) 
  | PE_BLSL(a,b) -> Int32.shift_left (eval_pe a) (Int32.to_int (eval_pe b))
  | PE_BLSR(a,b) -> Int32.shift_right_logical (eval_pe a) (Int32.to_int (eval_pe b) )
  | PE_BASR(a,b) -> Int32.shift_right (eval_pe a) (Int32.to_int (eval_pe b) )

  | PE_If(p,t,e) -> if is_true (eval_pe p) then eval_pe t else eval_pe e

let eval_pe pe = 
  let res = eval_pe pe in 
  (if !debug_pe then log_and_print "Value [%s] = %ld\n" 
    (pe_to_str pe) res ) ;
  res 

(************************************************************************
 * Evaluate a TP2 predicate
 ************************************************************************)
let rec eval game p = match p with 
  | Pred_Expr(e) -> is_true (eval_pe e) 
  | Pred_True -> true 
  | Pred_And(p1,p2) -> (eval game p1) && (eval game p2) 
  | Pred_Or(p1,p2) -> (eval game p1) || (eval game p2)
  | Pred_Not(p1) -> not (eval game p1) 
  | Pred_File_MD5(f,s) -> 
      if file_exists f then begin
        let digest = Digest.file f in
        let hex = Digest.to_hex digest in
        log_only "File [%s] has MD5 checksum [%s]\n" f hex ; 
        (String.uppercase hex) = (String.uppercase s)
      end else begin
        log_only "WARNING: File [%s] not found for MD5 checksum.\n" f ;
        false
      end

  | Pred_File_Exists(f) -> 
      let filename = (Var.substitute_user_vars_only f) in
      file_size filename > 0  

  | Pred_File_Exists_In_Game(f) -> 
      let old_allow_missing = !Load.allow_missing in 
      Load.allow_missing := [] ; 
      let res = 
      (try 
        let a,b = split f in
        Load.skip_next_load_error := true; 
        let buff,path = Load.load_resource "FILE_EXISTS_IN_GAME" game true a b in
          (String.length buff > 0) 
      with _ -> false ) in
      Load.allow_missing := old_allow_missing ;
      res
  | Pred_File_Size(f,s) -> file_size (Var.substitute_user_vars_only f) = s
  | Pred_File_Contains(f,r) -> 
      let buf = load_file f in
      let regexp = Str.regexp_case_fold r in 
      try 
        let _ = Str.search_forward regexp buf 0 in 
        true
      with Not_found -> false

(************************************************************************
 * Uninstall STRSET 
 ************************************************************************)
let uninstall_strset game filename = 
  if (file_exists filename) then begin
    (try
      let buff = load_file filename in 
      let record : Load.str_set_record list = 
        Marshal.from_string buff 0 
      in 
      List.iter (fun (i,m,f) -> 
        if (i < 0 || i > Array.length game.Load.dialog) then begin
          log_only "WARNING: Cannot uninstall STRING_SET #%d, out of range 0 -- %d\n" i (Array.length game.Load.dialog) 
        end else begin 
          log_only "Un-SET_STRING #%d from %s back to %s\n"
            i 
            (Tlk.short_print game.Load.dialog.(i) 18)
            (Tlk.short_print m 18) ;
          game.Load.dialog.(i) <- m ;
          game.Load.dialog_mod <- true; 
          match game.Load.dialogf with
          | Some(a) -> a.(i) <- f ; game.Load.dialogf_mod <- true; 
          | None -> () 
        end 
      ) record
    with e -> 
      log_and_print "WARNING: Unable to uninstall STRING_SET references from [%s]: %s\n" filename (Printexc.to_string e));
    Unix.unlink filename ;
  end else log_only "[%s] SET_STRING uninstall info not found\n" filename 

let record_strset_uninstall_info game filename = 
  if (game.Load.str_sets = []) then
    ()
  else begin 
    try 
      let outchan = open_out_bin filename in 
      Marshal.to_channel outchan game.Load.str_sets [];
      close_out outchan ;
      game.Load.str_sets <- [] 
    with e -> 
      log_and_print "WARNING: Unable to write STRING_SET uninstall info to [%s]: %s\n" filename (Printexc.to_string e) 
  end 

let handle_recursive_tp2 = ref (fun s -> failwith "cannot handle recursive
TP2 invocation at this time")

(************************************************************************
 * Uninstall a TP2 component
 ************************************************************************)
let handle_at_uninstall m do_uninstall do_interactive_uninstall = 
    List.iter (fun a -> match a with
      | TP_At_Uninstall(str) -> 
        if do_uninstall then begin
        let str = Var.substitute_user_vars_only str in 
        match (split (String.uppercase str)) with
        | _,"TP2" -> (!handle_recursive_tp2) str
        | _,_ -> let str = Arch.handle_view_command str in 
               ignore (Unix.system(Arch.slash_to_backslash str))
        end
      | TP_At_Interactive_Uninstall(str) -> 
        if do_interactive_uninstall then begin
          let str = Var.substitute_user_vars_only str in 
          match (split (String.uppercase str)) with
          | _,"TP2" -> (!handle_recursive_tp2) str
          | _,_ -> let str = Arch.handle_view_command str in 
                 ignore(Unix.system(Arch.slash_to_backslash str))
        end 
      | _ -> () 
    ) m.mod_parts  

let one_newline_regexp = Str.regexp "[\n]" 
let one_newline_or_cr_regexp = Str.regexp "[\r\n]" 
let many_newline_or_cr_regexp = Str.regexp "[\r\n]+" 
let many_whitespace_regexp = Str.regexp "[ \t]+" 
let many_not_whitespace_regexp = Str.regexp "[^ \t\n\r]+"

let uninstall_tp2_component game tp2 tp_file i interactive =
  try 
    let result = tp2 in
    let d = result.backup ^ "/" ^ (string_of_int i) in 
    let u_filename = (Printf.sprintf "%s/UNINSTALL.%d" d i) in 
    let u_strset_filename = (Printf.sprintf "%s/UNSETSTR.%d" d i) in 
    uninstall_strset game u_strset_filename ; 
    let buff = load_file u_filename in 
    let file_list = Str.split one_newline_regexp buff in
    log_and_print "Will uninstall %3d files for [%s] component %d.\n" 
      (List.length file_list) tp_file i; 
    List.iter (fun override_filename -> 
      let base = Filename.basename override_filename in 
      let backup_filename = d ^ "/" ^ base in 
      (* did we save a backup copy? *) 
      if file_exists backup_filename then begin
        (* copy it from Backup to Override *) 
        log_or_print "  Restoring backed-up [%s]\n" backup_filename ; 
        let buff = load_file backup_filename in
        let restore_chn = open_out_bin override_filename in
        output_string restore_chn buff ;
        close_out restore_chn ;
        Unix.unlink backup_filename 
      end else begin
        (* no backup copy, so just delete it from override *) 
        log_or_print "  Deleting [%s] (to restore original game condition)\n" 
          override_filename ; 
        try 
          Unix.unlink override_filename  
        with e ->
          log_or_print "Unable to Unlink [%s]: %s"
            override_filename (Printexc.to_string e)
      end 
    ) file_list  ;
    Unix.unlink u_filename ;
    let m = get_nth_module result i in 
    handle_at_uninstall m true interactive ;
    log_and_print "Uninstalled    %3d files for [%s] component %d.\n" 
      (List.length file_list) tp_file i; 
  with e ->
    log_and_print "Error Uninstalling [%s] component %d:\n%s\n"
      tp_file i (Printexc.to_string e)

(************************************************************************
 * Determine what has been installed. 
 ************************************************************************)
type status = Installed | Temporarily_Uninstalled | Permanently_Uninstalled
type installed_mods = (string * int * int * (string option) * status) list
  (* module : language : component : status *) 
let log_name = "WeiDU.log"
let the_log = ref [] 

let str_of_str_opt sopt = match sopt with
    Some(str) -> "~" ^ str ^ "~"
  | None -> ""

let print_log () = 
  List.iter (fun (n,i1,i2,sopt,st) ->
    log_or_print "%s %2d %2d %s %s\n" n i1 i2 (match st with
      Installed -> "Installed"
    | Temporarily_Uninstalled -> "Temporarily_Uninstalled"
    | Permanently_Uninstalled -> "Permanently_Uninstalled")
    (str_of_str_opt sopt) 
  ) !the_log

let save_log game handle_tp2_filename handle_tra_filename =
  let out = open_out log_name in
  log_or_print "Saving This Log:\n" ; print_log () ; 
  Printf.fprintf out "// Log of Currently Installed WeiDU Mods\n" ;
  Printf.fprintf out "// The top of the file is the 'oldest' mod\n" ; 
  Printf.fprintf out "// ~TP2_File~ #language_number #component_number // Component Name\n" ;
  let newline_regexp = one_newline_or_cr_regexp in 
  let my_ht = Hashtbl.create 511 in 
  List.iter (fun (a,b,c,sopt,d) -> 
    let str = 
      let component_name = 
        try
          (* log_or_print "*** Looking Up %s.\n" a; *)
          let tp2 = 
            try Hashtbl.find my_ht a 
            with _ -> 
              (* log_or_print "*** Parsing %s.\n" a; *)
              let res = handle_tp2_filename a in
              Hashtbl.add my_ht a res ; res
          in 
          Dc.clear_state () ;
          Dc.push_trans ();
          let a_dir = Filename.dirname a in 
          (try
            let l = List.nth tp2.languages b in
            List.iter (fun s -> 
              (* log_or_print "*** Loading %s for %s.\n" s a; *)
              handle_tra_filename s
            ) l.lang_tra_files ;
          with _ -> ()) ;
          let m = get_nth_module tp2 c in 
          let str = Dc.single_string_of_tlk_string game m.mod_name in 
          Dc.clear_state () ;
          Dc.pop_trans ();
          str 
        with _ -> 
          "???" 
      in 
      let component_name = Str.global_replace newline_regexp " " component_name in 
      Printf.sprintf "~%s~ #%d #%d // %s\n" 
        (String.uppercase a) b c component_name 
    in 
  match d with
    Installed -> output_string out str 
  | Temporarily_Uninstalled -> 
      log_and_print "Internal Error: saving a log with temporarily uninstalled module %s" str 
  | Permanently_Uninstalled ->
      Printf.fprintf out "// Recently Uninstalled: %s" str
  ) !the_log ; 
  close_out out 

let log_match a b = 
  Filename.basename (String.uppercase a) = Filename.basename (String.uppercase b) 

let already_installed tp2 i =
  let rec is_installed lst = match lst with
  | [] -> false
  | (a,b,c,sopt,d) :: tl when log_match a tp2 
        && c = i && d <> Permanently_Uninstalled -> true
  | hd :: tl -> is_installed tl 
  in is_installed !the_log 

let installed_lang_index tp2 =
  let rec is_installed lst = match lst with
  | [] -> None
  | (a,b,c,sopt,d) :: tl when log_match a tp2 
        && d <> Permanently_Uninstalled -> Some(b)
  | hd :: tl -> is_installed tl 
  in is_installed !the_log 

let temporarily_uninstalled tp2 i =
  let rec is_installed lst = match lst with
  | [] -> false
  | (a,b,c,sopt,d) :: tl when log_match a tp2
        && c = i && d = Temporarily_Uninstalled -> true
  | hd :: tl -> is_installed tl 
  in is_installed !the_log 

let temp_to_perm_uninstalled tp2 i handle_tp2_filename = 
  let rec is_installed lst = match lst with
  | [] -> []
  | (a,b,c,sopt,d) :: tl when log_match a tp2 
        && c = i && d = Temporarily_Uninstalled ->
        (* if there were any "at_uninstall" actions here, do them! *) 
        let tp_file = a in 
        let tp2 = handle_tp2_filename tp_file in 
        let lang_name = 
        (try
          let l = List.nth tp2.languages b in
          l.lang_dir_name ; 
        with _ -> "" ) in 
        Var.add_var "LANGUAGE" lang_name ;
        let m = get_nth_module tp2 c in 
        log_only "Running AT_INTERACTIVE_EXITs in ~%s~ %d %d %s\n" 
          (String.uppercase a) b c
          (str_of_str_opt sopt) ;
        handle_at_uninstall m 
          false (* "AT_UNINSTALL" was already done! *)
          true (* but the user just asked for this to be explicit *) ;
        Var.remove_var "LANGUAGE" ;
        (a,b,c,sopt,Permanently_Uninstalled) :: tl 
  | hd :: tl -> hd :: (is_installed tl)
  in the_log := is_installed !the_log 

(************************************************************************
 * Do everything necessary to uninstall the given tp2 component. This
 * may require temporarily uninstalling stuff that has been installed
 * since our target was installed. 
 ************************************************************************)
let rec find_best_file lst =
  match lst with
    hd :: tl -> if file_exists hd then hd 
                else if file_exists (Filename.basename hd) then
                  (Filename.basename hd)
                else find_best_file tl
  | [] -> failwith "TP2 not found!"  

let uninstall game handle_tp2_filename tp2 i interactive =
  log_or_print "uninstall: %s %d\n" tp2 i ; 
  let worked = ref true in 
  if not (already_installed tp2 i) then begin
    log_and_print "Internal Error: trying to uninstall non-installed mod component [%s] %d\n" tp2 i ;
    false 
  end else if (temporarily_uninstalled tp2 i) then begin
    log_or_print "uninstall: %s %d already temporarily uninstalled\n" tp2 i ;
    temp_to_perm_uninstalled tp2 i handle_tp2_filename ; 
    true 
  end else begin
    let rec prepare lst = match lst with
    | [] -> [] (* end of the line *)

    (* this is the entry in the list *)
    | (a,b,c,sopt,d) :: tl when log_match a tp2 && c = i -> 
      begin match d with
      | Permanently_Uninstalled -> (* some sort of error *) 
                  log_and_print "Internal Error: mod component [%s] %d already uninstalled\n" tp2 i ; (a,b,c,sopt,Permanently_Uninstalled) :: tl
      | Temporarily_Uninstalled -> (* we just won't restore it! *)
                      (a,b,c,sopt,Permanently_Uninstalled) :: tl
      | Installed -> 
          begin 
          try 
            let best = find_best_file [a ; tp2] in 
            uninstall_tp2_component game (handle_tp2_filename best) a c interactive ;
            (a,b,c,sopt,Permanently_Uninstalled) :: tl
          with _ ->
            log_and_print "ERROR: This Mod is too old (or too new) to uninstall that component for you.\nUpgrade to the newest versions of this mod and that one and try again.\n" ;
            worked := false ; 
            lst 
          end 
      end

    | (a,b,c,sopt,d) :: tl -> 
      begin match d with
      | Permanently_Uninstalled    
      | Temporarily_Uninstalled -> (* keep going *)
                      (a,b,c,sopt,d) :: (prepare tl)
      | Installed -> 
          log_or_print "We must temporarily uninstall [%s] component %d\n" 
            a c ; 
          begin
          try
            let best = find_best_file [ a] in 
            uninstall_tp2_component game (handle_tp2_filename best) a c false ; 
            (* take away for now *)
            (a,b,c,sopt,Temporarily_Uninstalled) :: (prepare tl)
          with e -> 
            log_and_print "ERROR: This Mod is too old (or too new) to uninstall that component for you.\nUpgrade to the newest versions of this mod and that one and try again.\n[%s]\n" 
            (Printexc.to_string e);
            worked := false ; 
            lst 
          end
      end
    in 
    let new_log = List.rev (prepare (List.rev !the_log)) in
    the_log := new_log ;
    !worked
  end

let body_of_script buff = 
  if buff = "" then "" else 
  try 
    let first_nl = String.index buff '\n' in
    let last_nl = (String.rindex buff 'S') - 1 in 
    if first_nl = last_nl then 
      "" 
    else 
      let length = (last_nl - first_nl) - 1 in 
      String.sub buff (first_nl + 1) length 
  with e -> 
    log_and_print "ERROR: not a BCS script\n" ;
    failwith "not a BCS script" 

type default_action = TP_Install | TP_Uninstall | TP_Skip | TP_Ask 

(************************************************************************
 * change location "where" in "buff" to point to str-ref "what"
 ************************************************************************)
let process_patch1 patch_filename game buff p = 
  let bounds_check idx size = 
    let len = String.length buff in 
    if (idx < 0 || (idx + size) > len) then begin 
      log_and_print "ERROR: illegal %d-byte read from offset %d of %d-byte file %s\n" size idx len patch_filename ;
      failwith (patch_filename ^ ": read out of bounds")
    end 
  in 
  match p with 
  | TP_PatchReadByte(where,name) ->
      let where = Int32.to_int (eval_pe where) in 
      bounds_check where 1 ; 
      let value = byte_of_str_off buff where in
      Var.add_var name (Printf.sprintf "%d" value)
  | TP_PatchReadShort(where,name) ->
      let where = Int32.to_int (eval_pe where) in 
      bounds_check where 2 ; 
      let value = short_of_str_off buff where in
      Var.add_var name (Printf.sprintf "%d" value) 
  | TP_PatchReadLong(where,name) ->
      let where = Int32.to_int (eval_pe where) in 
      bounds_check where 3 ; 
      let value = int32_of_str_off buff where in
      Var.add_var name (Printf.sprintf "%ld" value) 
  | TP_PatchReadAscii(where,name) ->
      let where = Int32.to_int (eval_pe where) in
      bounds_check where 8 ; 
      let value = get_string_of_size buff where 8 in
      Var.add_var name (Printf.sprintf "%s" value) 
  | TP_Read2DA(row, col, req_col, var_name) -> 
      let row = Int32.to_int (eval_pe row) in
      let col = Int32.to_int (eval_pe col) in
      let req_col = Int32.to_int (eval_pe req_col) in 
      let lines = Str.split many_newline_or_cr_regexp buff in
      let entries = List.map (fun line -> 
        let entry_list = Str.split many_whitespace_regexp line in
        Array.of_list entry_list) lines in
      let rec process line_list lines_left = 
        match line_list, lines_left with
          [], 0 -> ()
        | [], _ -> 
          begin
            log_and_print "ERROR: Cannot find %d rows with at least %d columns." row req_col ;
            failwith "Cannot Read 2DA Entry" 
          end
        | hd :: tl, i  when Array.length hd < req_col -> process tl i 
        | hd :: tl, 0 -> Var.add_var var_name hd.(col) 
        | hd :: tl, i -> process tl (i - 1)
      in
      process entries row ;

  | _ -> () 


let rec handle_tp 
      game 
      handle_tp2_filename
      handle_tra_filename 
      handle_d_filename
      compile_baf_filename
      handle_script_buffer
      handle_dlg_buffer
      handle_d_buffer
      enqueue_tp2_filename
      emit_dlg_files
      this_tp2_filename
      tp 
= begin

let rec process_patch2 patch_filename game buff p =
  process_patch1 patch_filename game buff p ; 
  match p with
    TP_PatchStrRef(where,what) -> 
      let where = Int32.to_int (eval_pe where) in 
      let new_index = match Dc.resolve_tlk_string game what with
        Dlg.TLK_Index(i) -> i
      | _ -> log_and_print "ERROR: cannot resolve SAY patch\n" ; failwith "resolve" 
      in 
      let new_string = str_of_int new_index in 
      String.blit new_string 0 buff where 4 ;
      buff

  | TP_PatchReplaceBCSBlock(old_file, new_file) -> begin
      let bcs_buff_of_baf_or_bcs file =
        let a,b = split (String.uppercase file) in 
        if b = "BAF" then begin
          try 
            let script = parse_file file "parsing .baf files" 
              (Bafparser.baf_file Baflexer.initial) in 
            let buff = Buffer.create 1024 in 
            Bcs.save_bcs game (Bcs.Save_BCS_Buffer(buff)) script ;
            Buffer.contents buff  
          with e -> 
            log_and_print "ERROR: error compiling [%s]: %s\n" 
              file (Printexc.to_string e) ;
            raise e
        end else begin
          load_file file
        end 
      in 
      let old_file_buff = bcs_buff_of_baf_or_bcs old_file in  
      let string_to_find = body_of_script old_file_buff in
      let new_file_buff = bcs_buff_of_baf_or_bcs new_file in  
      let string_to_sub_in = body_of_script new_file_buff in
      let my_regexp = Str.regexp_string string_to_find in
      try 
        let _ = Str.search_forward my_regexp buff 0 in
        Str.global_replace my_regexp string_to_sub_in buff 
      with Not_found -> 
        log_and_print "WARNING: cannot find block matching [%s]\n"
          old_file ;
        buff
      end 

  | TP_PatchReplaceBCSBlockRE(old_file, new_file) -> begin
      let string_to_find = load_file old_file in
      let string_to_sub_in = load_file new_file in
      let my_regexp = Str.regexp string_to_find in
      try 
        let _ = Str.search_forward my_regexp buff 0 in
        Str.global_replace my_regexp string_to_sub_in buff 
      with Not_found -> 
        log_and_print "WARNING: cannot find block matching [%s]\n"
          old_file ;
        buff
      end 

  | TP_PatchApplyBCSPatch(patch_file,opt_overwrite) -> begin
      let patch_buff = load_file patch_file in
      try
        let new_buff, bad_chunks, app_chunks = 
            Diff.do_patch buff patch_buff false in begin
          if ( bad_chunks > 0 ) then begin
            log_or_print 
              "ERROR: Cannot apply patch %s (%d bad chunks).\n" 
              patch_file bad_chunks ;
            match opt_overwrite with
            | Some(newfile) -> load_file newfile 
            | None -> failwith "Cannot Apply Patch"
          end else begin 
          if ( app_chunks > 0 ) then begin
            log_or_print "WARNING: %d chunks in patch file %s already applied.\n" app_chunks patch_file 
          end ;
          new_buff
          end 
        end
      with Not_found ->
        log_and_print "Error: applying patch %s failed.\n" patch_file ;
        buff
  end

  | TP_PatchInsertBytes(where,how_many) ->
      let how_many = Int32.to_int (eval_pe how_many) in 
      let blanks = String.make how_many ('\000') in
      let where = Int32.to_int (eval_pe where) in 
      let before = Str.string_before buff where in
      let after = Str.string_after buff where in
      before ^ blanks ^ after

  | TP_PatchDeleteBytes(where,how_many) ->
      let where = Int32.to_int (eval_pe where) in 
      let before = Str.string_before buff where in
      let how_many = Int32.to_int (eval_pe how_many) in 
      let after = Str.string_after buff (where + how_many) in
      before ^ after

  | TP_PatchSet(name,value) -> 
      let value = (eval_pe value) in
      Var.add_var name (Int32.to_string value) ;
      buff

  | TP_PatchIf(pred,tb,eb) -> 
      let b = ref buff in
      b := List.fold_left (fun acc elt -> 
          process_patch1 patch_filename game acc elt; 
          process_patch2 patch_filename game acc elt) !b 
          (if is_true (eval_pe pred) then tb else eb) ;
      !b

  | TP_PatchWhile(pred,pl) -> 
      let b = ref buff in
      while is_true (eval_pe pred) do
        b := List.fold_left (fun acc elt -> 
            process_patch1 patch_filename game acc elt; 
            process_patch2 patch_filename game acc elt) !b pl 
      done ;
      !b
  | TP_PatchFor(init,guard,inc,body) -> 
      let cmd_list = init @ [ TP_PatchWhile(guard,body @ inc) ] in
      let b = ref buff in
      b := List.fold_left (fun acc elt -> 
          process_patch1 patch_filename game acc elt; 
          process_patch2 patch_filename game acc elt) !b cmd_list  ;
      !b

  | TP_Patch2DA(row,col,req_col,value) ->
      let row = Int32.to_int (eval_pe row) in
      let col = Int32.to_int (eval_pe col) in
      let req_col = Int32.to_int (eval_pe req_col) in 
      let lines = Str.split many_newline_or_cr_regexp buff in

      eval_pe_warn := false ; 
      let value = 
        try 
          let x = (eval_pe value) in
          Int32.to_string x 
        with _ -> 
          begin
            match value with
            | PE_String(x) -> x
            | _ -> (eval_pe_warn := true ; ignore (eval_pe value) ; "")
          end 
      in 
      let slv = String.length value in 
      let max = ref slv in 
      let entries = List.map (fun line -> 
        let entry_list = Str.split many_whitespace_regexp line in
        List.iter (fun e ->
          let len = String.length e in
          if len > !max then max := len
        ) entry_list ; 
        Array.of_list entry_list) lines 
      in
      let newlines_sofar = ref 0 in 
      let rec process line_list lines_left = 
        match line_list, lines_left with
          [], 0 -> ()
        | [], _ -> 
          begin
            log_and_print "ERROR: Cannot find %d rows with at least %d columns." row req_col ;
            failwith "Cannot Set 2DA Entry" 
          end
        | hd :: tl, i  when Array.length hd < req_col -> 
            incr newlines_sofar ; 
            process tl i 
        | hd :: tl, 0 -> 
            hd.(col) <- value 
        | hd :: tl, i -> begin
            incr newlines_sofar ;
            process tl (i - 1)
            end 
      in
      process entries row ;
      (try begin 
        let pos = ref 0 in 
        for n = 1 to !newlines_sofar do
          pos := (Str.search_forward many_newline_or_cr_regexp buff !pos) ;
          pos := Str.match_end () ;
        done ;
        for n = 1 to col do
          pos := (Str.search_forward many_whitespace_regexp buff !pos) ;
          pos := Str.match_end () ; 
        done ;
        let before = !pos in 
        let after = Str.search_forward many_not_whitespace_regexp buff !pos in
        let after = Str.match_end () in 
        let before_str = Str.string_before buff before in 
        let after_str = Str.string_after buff after in 
        Printf.sprintf "%s%s%s" before_str value after_str 
      end with _ -> begin 
        log_or_print "WARNING: Fast SET_2DA_ENTRY failed, falling back on old method...\n" ;
        let buf = Buffer.create (slv * 2) in 
        List.iter (fun entry_array ->
          Array.iter (fun elt -> 
            Printf.bprintf buf "%-.*s\t" !max elt
          ) entry_array ;
          Buffer.add_char buf '\n' 
        ) entries ; 
        Buffer.contents buf 
        end )
      (*
      List.fold_left (fun line_acc entry_array -> 
        line_acc ^ (Array.fold_left (fun acc elt ->
          Printf.sprintf "%s%-.*s\t" acc !max elt) "" entry_array) ^ "\n") 
        "" entries
        *)

  | TP_PatchStringTextually(find,what) -> 
      let my_regexp = Str.regexp_case_fold find in
      let what = Var.substitute what in 
      Str.global_replace my_regexp what buff 

  | TP_PatchStringEvaluate(find, pl, replace) -> 
      (* REPLACE_EVALUATE ~Give(\([0-9]+\))~ 
                          SET "RESULT" = "%MATCH1%" 
                          ~Give(%RESULT%)~ 
         ( "%MATCH%" / 2 ) *)
      let my_regexp = Str.regexp find in
      let i = ref 0 in 
      let work_buff = ref buff in 
      (try 
        while true do
          let start_idx = Str.search_forward my_regexp !work_buff !i in 
          for j = 0 to 20 do
            let v = Printf.sprintf "MATCH%d" j in 
            Var.remove_var v ; 
            (try let group = Str.matched_group j !work_buff in 
                 Var.add_var v group ; 
             with _ -> ()) 
          done ; 
          (try 
            ignore (
              List.fold_left (fun acc elt -> 
                process_patch1 patch_filename game acc elt; 
                process_patch2 patch_filename game acc elt) !work_buff pl 
            ) ;
           with _ -> ());
          let this_replacement = Var.substitute_user_vars_only replace in
          let old_before = Str.string_before !work_buff start_idx in
          let old_after = Str.string_after !work_buff start_idx in 
          let new_after = 
            Str.replace_first my_regexp this_replacement old_after
          in 
          work_buff := (old_before ^ new_after) ;
          i := start_idx + 1
        done
      with _ -> () ) ;
      !work_buff

  | TP_PatchString(find,what) -> 
      let my_regexp = Str.regexp_case_fold find in
      let new_index = match Dc.resolve_tlk_string game what with
        Dlg.TLK_Index(i) -> i
      | _ -> log_and_print "ERROR: cannot resolve REPLACE patch\n" ; failwith "resolve" 
      in 
      let new_string = Printf.sprintf "%d" new_index in 
      Str.global_replace my_regexp new_string buff 

  | TP_PatchByte(where,what) -> 
      let where = Int32.to_int (eval_pe where) in 
      let what = Int32.to_int (eval_pe what) in 
      let str = String.make 1 (Char.chr what) in
      String.blit str 0 buff where 1 ;
      buff
  | TP_PatchShort(where,what) -> 
      let where = Int32.to_int (eval_pe where) in 
      let what = Int32.to_int (eval_pe what) in 
      let str = str_of_short what in 
      String.blit str 0 buff where 2 ;
      buff
  | TP_PatchLong(where,what) -> 
      let where = Int32.to_int (eval_pe where) in 
      let what = eval_pe what in 
      let str = str_of_int32 what in 
      String.blit str 0 buff where 4 ;
      buff
  | TP_PatchWriteFile(where,filename,insert) ->  
      let where = Int32.to_int (eval_pe where) in 
      let file_buff = load_file filename in 
      if insert then begin
        let before = Str.string_before buff where in
        let after = Str.string_after buff where in
        before ^ file_buff ^ after 
      end else begin
        let file_len = String.length file_buff in 
        let len = String.length buff in  
        if (len - where) < file_len then begin 
          log_and_print "Not enough room for [%s] (%d bytes) if you start at position %d in a %d byte buffer!" (filename) (file_len) (where) (len) ;
          failwith "ERROR: cannot process WRITE_FILE" 
        end ;
        String.blit file_buff 0 buff where file_len ;
        buff
      end 

  | TP_PatchASCII(where,what,evaluate) -> 
      let where = Int32.to_int (eval_pe where) in 
      let what = if evaluate then Var.substitute_user_vars_only what 
                             else what in 
      String.blit what 0 buff where (String.length what) ;
      buff

  | TP_Patch_Gam(cre_name,area,x,y) ->

        (* Patch baldur.gam in save directories *)

        let dlist = list_of_files_in_directory "save" in
        List.iter
        (fun filename ->
        let filename = "save/" ^ filename in
        if is_directory filename && filename <> "save/." && filename <> "save/.." then begin
        log_and_print "Patching baldur.gam in directory %s\n" filename;
        let baldur_buff = load_file (filename ^ "/baldur.gam") in
        if String.sub baldur_buff 0 8 <> "GAMEV1.1" then begin
        failwith "not a valid GAME V1.1 file (wrong sig)"
        end ;
        let non_party_npc_off = int_of_str_off baldur_buff 0x30 in
        let num_non_party_npc = int_of_str_off baldur_buff 0x34 in
        let variable_off = int_of_str_off baldur_buff 0x38 in
        let journal_off = int_of_str_off baldur_buff 0x50 in
        let last_npc_off = (non_party_npc_off + ((num_non_party_npc) * 352)) in

        (* Create the non-party NPC entry *)

        let npc_entry_buff = String.make 352 '\000' in
        write_short npc_entry_buff 0x2 0xFFFF;
        write_int npc_entry_buff 0x4 (variable_off + 352);
        write_int npc_entry_buff 0x8 (String.length buff);
        String.blit area 0 npc_entry_buff 0x18 (String.length area);
        write_short npc_entry_buff 0x20 x;
        write_short npc_entry_buff 0x22 y;

        (* Update the offsets *)

        write_int baldur_buff 0x38 (variable_off + (String.length buff) + 352);
        write_int baldur_buff 0x50 (journal_off + (String.length buff) + 352);
        write_int baldur_buff 0x34 (num_non_party_npc + 1);

        (* Splice in the non-party NPC entry and the cre file *)

        for i = 0 to num_non_party_npc - 1 do
        let cre_off = int_of_str_off baldur_buff (non_party_npc_off + (i * 352) + 0x4) in
        write_int baldur_buff (non_party_npc_off + (i * 352) + 0x4) (cre_off + 352)
        done;

        let before = Str.string_before baldur_buff last_npc_off in
        let after = Str.string_after baldur_buff variable_off in
        let cre_chunk = String.sub baldur_buff last_npc_off (variable_off - last_npc_off) in
        let new_buff = before ^ npc_entry_buff ^ cre_chunk ^ buff ^ after in

        let oc = open_out_bin (filename ^ "/baldur.gam") in
        output_string oc new_buff;
        close_out oc;
        end) dlist;

        (* Patch baldur.gam in mpsave directories *)

        let mpdlist = list_of_files_in_directory "mpsave" in
        List.iter
        (fun filename ->
        let filename = "mpsave/" ^ filename in
        if is_directory filename && filename <> "mpsave/." && filename <> "mpsave/.." then begin
        log_and_print "Patching baldur.gam in directory %s\n" filename;
        let baldur_buff = load_file (filename ^ "/baldur.gam") in
        if String.sub baldur_buff 0 8 <> "GAMEV1.1" then begin
        failwith "not a valid GAME V1.1 file (wrong sig)"
        end ;
        let non_party_npc_off = int_of_str_off baldur_buff 0x30 in
        let num_non_party_npc = int_of_str_off baldur_buff 0x34 in
        let variable_off = int_of_str_off baldur_buff 0x38 in
        let journal_off = int_of_str_off baldur_buff 0x50 in
        let last_npc_off = (non_party_npc_off + ((num_non_party_npc) * 352)) in

        (* Create the non-party NPC entry *)

        let npc_entry_buff = String.make 352 '\000' in
        write_short npc_entry_buff 0x2 0xFFFF;
        write_int npc_entry_buff 0x4 (variable_off + 352);
        write_int npc_entry_buff 0x8 (String.length buff);
        String.blit area 0 npc_entry_buff 0x18 (String.length area);
        write_short npc_entry_buff 0x20 x;
        write_short npc_entry_buff 0x22 y;

        (* Update the offsets *)

        write_int baldur_buff 0x38 (variable_off + (String.length buff) + 352);
        write_int baldur_buff 0x50 (journal_off + (String.length buff) + 352);
        write_int baldur_buff 0x34 (num_non_party_npc + 1);

        (* Splice in the non-party NPC entry and the cre file *)

        for i = 0 to num_non_party_npc - 1 do
        let cre_off = int_of_str_off baldur_buff (non_party_npc_off + (i * 352) + 0x4) in
        write_int baldur_buff (non_party_npc_off + (i * 352) + 0x4) (cre_off + 352)
        done;

        let before = Str.string_before baldur_buff last_npc_off in
        let after = Str.string_after baldur_buff variable_off in
        let cre_chunk = String.sub baldur_buff last_npc_off (variable_off - last_npc_off) in
        let new_buff = before ^ npc_entry_buff ^ cre_chunk ^ buff ^ after in

        let oc = open_out_bin (filename ^ "/baldur.gam") in
        output_string oc new_buff;
        close_out oc;
        end) mpdlist;

        (* Patch the baldur.gam in the biff *)

        log_and_print "Patching starting baldur.gam (for new games)...\n";
        let nbaldur_buff,path = Load.load_resource "PATCH_GAM" game true "BALDUR" "GAM" in

        if String.sub nbaldur_buff 0 8 <> "GAMEV1.1" then begin
        failwith "not a valid GAME V1.1 file (wrong sig)"
        end ;
        let party_npc_off = int_of_str_off nbaldur_buff 0x20 in
        let journal_off = int_of_str_off nbaldur_buff 0x50 in
        let num_non_party_npc = int_of_str_off nbaldur_buff 0x34 in

        (* Create the non-party NPC entry *)

        let npc_entry_buff = String.make 352 '\000' in
        String.blit (String.uppercase cre_name) 0 npc_entry_buff 0xc (String.length cre_name);
        String.blit area 0 npc_entry_buff 0x18 (String.length area);
        write_short npc_entry_buff 0x20 x;
        write_short npc_entry_buff 0x22 y;
        for i = 140 to 147 do
        npc_entry_buff.[i] <- Char.chr 0xFF
        done;
        for i = 180 to 185 do
        npc_entry_buff.[i] <- Char.chr 0xFF
        done;

        (* Update all offsets *)

        write_int nbaldur_buff 0x20 (party_npc_off + 352);
        write_int nbaldur_buff 0x28 (party_npc_off + 352);
        write_int nbaldur_buff 0x50 (journal_off + 352);
        write_int nbaldur_buff 0x34 (num_non_party_npc + 1);

        (* Splice in the non-party NPC entry *)

        let before = Str.string_before nbaldur_buff party_npc_off in
        let after = Str.string_after nbaldur_buff party_npc_off in
        let new_buff = before ^ npc_entry_buff ^ after in

        let oc = open_out_bin "override/baldur.gam" in
        output_string oc new_buff;
        close_out oc;
        buff

  | TP_Remove_Known_Spell(sp_list) ->

      (* Check for version *)

      let cre_v = String.sub buff 0 8 in
      (match cre_v with
      | "CRE V1.0" ->

        let knownspelloff = int_of_str_off buff 0x2a0 in
        let spellmemoff = int_of_str_off buff 0x2a8 in
        let numknownspells = int_of_str_off buff 0x2a4 in

        (* Grab all spells *)

        let spell_list = ref [] in
        let spell_string = String.sub buff knownspelloff (spellmemoff - knownspelloff) in
        for i = 0 to numknownspells - 1 do
        let spell_check = String.sub spell_string (i * 12) 8 in
        let spell_check = get_string_of_size spell_check 0 8 in
        spell_list := (String.uppercase spell_check) :: !spell_list
        done;

        spell_list := List.rev !spell_list;

        let i = ref 0 in
        let match_count = ref 0 in
        let work_buff = ref buff in
        List.iter (fun spell_name ->
        if List.mem spell_name sp_list then begin
        let spellmemoff = int_of_str_off !work_buff 0x2a8 in
        let memspelloff = int_of_str_off !work_buff 0x2b0 in
        let itemsslotoff = int_of_str_off !work_buff 0x2b8 in
        let itemsoff = int_of_str_off !work_buff 0x2bc in
        let effoff = int_of_str_off !work_buff 0x2c4 in
        let numknownspells = int_of_str_off !work_buff 0x2a4 in
        write_int !work_buff 0x2a8 (spellmemoff - 12);
        write_int !work_buff 0x2b0 (memspelloff - 12);
        write_int !work_buff 0x2b8 (itemsslotoff - 12);
        write_int !work_buff 0x2bc (itemsoff - 12);
        write_int !work_buff 0x2c4 (effoff - 12);
        write_int !work_buff 0x2a4 (numknownspells - 1);
        let before_buff = Str.string_before !work_buff (knownspelloff + ((!i - !match_count) * 12)) in
        let after_buff = Str.string_after !work_buff (knownspelloff + ((!i - !match_count) * 12) + 12) in
        work_buff := before_buff ^ after_buff ;
        incr i ;
        incr match_count
        end else
        incr i) !spell_list ;

        !work_buff

      | "CRE V9.0" ->

        let knownspelloff = int_of_str_off buff 0x308 in
        let spellmemoff = int_of_str_off buff 0x310 in
        let numknownspells = int_of_str_off buff 0x30c in

        (* Grab all spells *)

        let spell_list = ref [] in
        let spell_string = String.sub buff knownspelloff (spellmemoff - knownspelloff) in
        for i = 0 to numknownspells - 1 do
        let spell_check = String.sub spell_string (i * 12) 8 in
        let spell_check = get_string_of_size spell_check 0 8 in
        spell_list := (String.uppercase spell_check) :: !spell_list
        done;

        spell_list := List.rev !spell_list;

        let i = ref 0 in
        let match_count = ref 0 in
        let work_buff = ref buff in
        List.iter (fun spell_name ->
        if List.mem spell_name sp_list then begin
        let spellmemoff = int_of_str_off !work_buff 0x310 in
        let memspelloff = int_of_str_off !work_buff 0x318 in
        let itemsslotoff = int_of_str_off !work_buff 0x320 in
        let itemsoff = int_of_str_off !work_buff 0x324 in
        let effoff = int_of_str_off !work_buff 0x32c in
        let numknownspells = int_of_str_off !work_buff 0x30c in
        write_int !work_buff 0x310 (spellmemoff - 12);
        write_int !work_buff 0x318 (memspelloff - 12);
        write_int !work_buff 0x320 (itemsslotoff - 12);
        write_int !work_buff 0x324 (itemsoff - 12);
        write_int !work_buff 0x32c (effoff - 12);
        write_int !work_buff 0x30c (numknownspells - 1);
        let before_buff = Str.string_before !work_buff (knownspelloff + ((!i - !match_count) * 12)) in
        let after_buff = Str.string_after !work_buff (knownspelloff + ((!i - !match_count) * 12) + 12) in
        work_buff := before_buff ^ after_buff ;
        incr i ;
        incr match_count
        end else
        incr i) !spell_list ;

        !work_buff

     | _ -> failwith "ERROR: REMOVE_KNOWN_SPELL: Unknown cre version %s" cre_v);

  | TP_Add_Known_Spell(spell,level,sp_type) ->
      
      (* Create the new known spell *)

      let spell_buff = String.make 12 '\000' in
      String.blit spell 0 spell_buff 0 (String.length spell);
      write_short spell_buff 8 level;

      let new_type =
      (match (String.uppercase sp_type) with
      | "PRIEST" -> 0
      | "INNATE" -> 2
      | "WIZARD" -> 1
      | _ -> log_and_print "WARNING: ADD_NEW_SPELL: Unknown flag %s.  Defaulting to INNATE for spell type.\n" (String.uppercase sp_type);
             2)
      in

      write_short spell_buff 10 new_type ;

      (* Check for version *)
      
      let cre_v = String.sub buff 0 8 in
      (match cre_v with
      | "CRE V1.0" ->

        let spellmemoff = int_of_str_off buff 0x2a8 in
        let memspelloff = int_of_str_off buff 0x2b0 in
        let itemsslotoff = int_of_str_off buff 0x2b8 in
        let itemsoff = int_of_str_off buff 0x2bc in
        let effoff = int_of_str_off buff 0x2c4 in
        write_int buff 0x2a8 (spellmemoff + 12);
        write_int buff 0x2b0 (memspelloff + 12);
        write_int buff 0x2b8 (itemsslotoff + 12);
        write_int buff 0x2bc (itemsoff + 12);
        write_int buff 0x2c4 (effoff + 12);
        let knownspelloff = int_of_str_off buff 0x2a0 in
        let numknownspells = int_of_str_off buff 0x2a4 in
        write_int buff 0x2a4 (numknownspells + 1);
        let before_buff = Str.string_before buff knownspelloff in
        let after_buff = Str.string_after buff knownspelloff in
        before_buff ^ spell_buff ^ after_buff

     | "CRE V9.0" ->

        let spellmemoff = int_of_str_off buff 0x310 in
        let memspelloff = int_of_str_off buff 0x318 in
        let itemsslotoff = int_of_str_off buff 0x320 in
        let itemsoff = int_of_str_off buff 0x324 in
        let effoff = int_of_str_off buff 0x32c in
        write_int buff 0x310 (spellmemoff + 12);
        write_int buff 0x318 (memspelloff + 12);
        write_int buff 0x320 (itemsslotoff + 12);
        write_int buff 0x324 (itemsoff + 12);
        write_int buff 0x32c (effoff + 12);
        let knownspelloff = int_of_str_off buff 0x308 in
        let numknownspells = int_of_str_off buff 0x30c in
        write_int buff 0x30c (numknownspells + 1);
        let before_buff = Str.string_before buff knownspelloff in
        let after_buff = Str.string_after buff knownspelloff in
        before_buff ^ spell_buff ^ after_buff

     | _ -> failwith "ERROR: ADD_NEW_SPELL: Unknown cre version %s" cre_v);

  | TP_Add_Map_Note(m) ->

	  (* Turn the colours into something usable *)

      let colour_flags =
      (match (String.uppercase m.colour) with
      | "GRAY" -> 0
      | "VIOLET" -> 1
      | "GREEN" -> 2
      | "ORANGE" -> 3
      | "RED" -> 4
      | "BLUE" -> 5
      | "DARKBLUE" -> 6
      | "LIGHTGRAY" -> 7
      | _ -> log_and_print "WARNING: ADD_MAP_NOTE: Unknown flag %s.  Defaulting to GRAY for flags.\n" (String.uppercase m.colour);
             0) in

	  (* Create the new automap *)

      let auto_buff = String.make 52 '\000' in
      write_short auto_buff 0 m.xcoord;
      write_short auto_buff 2 m.ycoord;
      write_byte auto_buff 8 1;
      write_byte auto_buff 10 colour_flags;

      let new_index = match Dc.resolve_tlk_string game m.mstring with
   	   Dlg.TLK_Index(i) -> i
      | _ -> log_and_print "ERROR: ADD_MAP_NOTE: cannot resolve SAY patch\n" ; failwith "resolve"
      in
      let new_string = str_of_int new_index in
      String.blit new_string 0 auto_buff 4 4 ;

      (* Check for version *)

  	  let area_v = String.sub buff 0 8 in
	  (match area_v with
       | "AREAV1.0" ->
         let automap_off = int_of_str_off buff 0xc4 in
		 let auto_num = int_of_str_off buff 0xc8 in
	     if automap_off = 0 then begin
         let new_automap_off = String.length buff in
		 write_int buff 0xc4 new_automap_off;
         write_int buff 0xc8 (auto_num + 1)
		 end else
         write_int buff 0xc8 (auto_num + 1)
       | "AREAV9.1" ->
         let automap_off = int_of_str_off buff 0xd4 in
		 let auto_num = int_of_str_off buff 0xd8 in
	     if automap_off = 0 then begin
         let new_automap_off = String.length buff in
		 write_int buff 0xd4 new_automap_off;
         write_int buff 0xd8 (auto_num + 1)
		 end else
         write_int buff 0xd8 (auto_num + 1)
	   | _ -> failwith "ADD_MAP_NOTE: Unknown area version %s" area_v);

	  (* Splice it all together *)

      buff ^ auto_buff

  | TP_Add_Cre_Item(i) ->

      (* Create the new item *)

      let item_buff = String.make 20 '\000' in
      String.blit i.item_name 0 item_buff 0 (String.length i.item_name);
      write_short item_buff 0xa i.i_charge1;
      write_short item_buff 0xc i.i_charge2;
      write_short item_buff 0xe i.i_charge3;

      (* Make the item type into something usable *)

      let itype =
      (match (String.uppercase i.item_slot) with
      | "HELMET" -> 0
      | "ARMOR" -> 2
      | "SHIELD" -> 4
      | "GLOVES" -> 6
      | "LRING" -> 8
      | "RRING" -> 10 
      | "AMULET" -> 12
      | "BELT" -> 14
      | "BOOTS" -> 16
      | "WEAPON1" -> 18
      | "WEAPON2" -> 20
      | "WEAPON3" -> 22
      | "WEAPON4" -> 24
      | "QUIVER1" -> 26
      | "QUIVER2" -> 28
      | "QUIVER3" -> 30
      | "QUIVER4" -> 32
      | "CLOAK" -> 34
      | "QITEM1" -> 36
      | "QITEM2" -> 38
      | "QITEM3" -> 40
      | "INV1" -> 42
      | "INV2" -> 44
      | "INV3" -> 46
      | "INV4" -> 48
      | "INV5" -> 50
      | "INV6" -> 52
      | "INV7" -> 54
      | "INV8" -> 56
      | "INV9" -> 58
      | "INV10" -> 60
      | "INV11" -> 62
      | "INV12" -> 64
      | "INV13" -> 66
      | "INV14" -> 68
      | "INV15" -> 70
      | "INV16" -> 72
      | _ -> log_and_print "WARNING: ADD_CRE_ITEM: Uknown flags %s.  Default to INV16 for placement.\n" (String.uppercase i.item_slot);
             70)
      in

      (* Make the flags into something useable *)

      let new_flags =
      (match (String.uppercase i.i_flags) with
      | "NONE" -> 0
      | "IDENTIFIED" -> 1
      | "UNSTEALABLE" -> 2
      | "STOLEN" -> 4
      | "UNDROPPABLE" -> 8
      | "IDENTIFIED&STOLEN" -> 5
      | "IDENTIFIED&UNSTEALABLE" -> 3
      | "IDENTIFIED&UNDROPPABLE" -> 9
      | "UNSTEALABLE&UNDROPPABLE" -> 10
      | "STOLEN&UNDROPPABLE" -> 12
      | "IDENTIFIED&STOLEN&UNDROPPABLE" -> 13
      | "IDENTIFIED&UNSTEALABLE&UNDROPPABLE" -> 11
      | _ -> log_and_print "WARNING: ADD_CRE_ITEM: Unknown flag %s.  Defaulting to NONE for flags.\n" (String.uppercase i.i_flags);
             0)
      in

      write_int item_buff 0x10 new_flags;

      (* Check for version *)

      let cre_v = String.sub buff 0 8 in
      (match cre_v with
      | "CRE V1.0" ->

        (* Read in the offsets we need to update and grab the plus state *)

        let islot_off = int_of_str_off buff 0x2b8 in
        let items_off = int_of_str_off buff 0x2bc in
        let num_items = int_of_str_off buff 0x2c0 in
        let possible_plus = i.twohanded_weapon in
        write_int buff 0x2b8 (islot_off + 20);
        write_int buff 0x2c0 (num_items + 1);

        (* Check and see if we want to equip the item *)

        if i.equip then begin


        (* If it's a two-handed weapon, empty the shield slot. *)

        if not possible_plus then begin
        let shield_slot = short_of_str_off buff (islot_off + 4) in
        write_empty_inv_slot buff (islot_off + 42) shield_slot ;
        write_short buff (islot_off + 4) 65535 ;
        let slot = short_of_str_off buff (islot_off + itype) in
        if slot <> 65535 then begin
        write_empty_inv_slot buff (islot_off + 42) slot;
        write_short buff (islot_off + itype) num_items;
        end else
        write_short buff (islot_off + itype) num_items;
        end else begin

        (* Not a two-handed weapon, just empty the slot and move any item
           in the slot to the inventory *)

        let slot = short_of_str_off buff (islot_off + itype) in
        if slot = 65535 then
        write_short buff (islot_off + itype) num_items
        else begin
	    let slot = short_of_str_off buff (islot_off + itype) in
        write_empty_inv_slot buff (islot_off + 42) slot ;
        write_short buff (islot_off + itype) num_items;
        end;
        end;
        end

        (* We're not equipping anything here, so just move any
           item that may be in the slot to inventory *)

        else begin
        let slot = short_of_str_off buff (islot_off + itype) in
        if slot <> 65535 then begin
        write_empty_inv_slot buff (islot_off + 42) slot;
        write_short buff (islot_off + itype) num_items
        end else
        write_short buff (islot_off + itype) num_items
        end;

        (* Check if it's a weapon and select the appropriate slot *)

        let regex = Str.regexp "WEAPON[1-4]" in
        if i.equip &&
        Str.string_match regex i.item_slot 0 then begin
        let i_num = String.sub i.item_slot 6 1 in
        let i_num = int_of_string i_num in
        write_short buff (islot_off + 76) (i_num - 1)
        end;

        (* Splice it all together *)

        let splice_off = items_off + (num_items * 20) in
        let before_buff = Str.string_before buff splice_off in
        let after_buff = Str.string_after buff splice_off in
        before_buff ^ item_buff ^ after_buff

      | "CRE V9.0" ->

        (* Read in the offsets we need to update and grab the plus state *)

        let islot_off = int_of_str_off buff 0x320 in
        let items_off = int_of_str_off buff 0x324 in
        let num_items = int_of_str_off buff 0x328 in
        let possible_plus = i.twohanded_weapon in
        write_int buff 0x320 (islot_off + 20);
        write_int buff 0x328 (num_items + 1);

        (* Check and see if we want to equip the item *)

        if i.equip then begin


        (* If it's a two-handed weapon, empty the shield slot. *)

        if not possible_plus then begin
        let shield_slot = short_of_str_off buff (islot_off + 4) in
        write_empty_inv_slot buff (islot_off + 42) shield_slot ;
        write_short buff (islot_off + 4) 65535 ;
        let slot = short_of_str_off buff (islot_off + itype) in
        if slot <> 65535 then begin
        write_empty_inv_slot buff (islot_off + 42) slot;
        write_short buff (islot_off + itype) num_items;
        end else
        write_short buff (islot_off + itype) num_items;
        end else begin

        (* Not a two-handed weapon, just empty the slot and move any item
           in the slot to the inventory *)

        let slot = short_of_str_off buff (islot_off + itype) in
        if slot = 65535 then
        write_short buff (islot_off + itype) num_items
        else begin
	    let slot = short_of_str_off buff (islot_off + itype) in
        write_empty_inv_slot buff (islot_off + 42) slot ;
        write_short buff (islot_off + itype) num_items;
        end;
        end;
        end

        (* We're not equipping anything here, so just move any
           item that may be in the slot to inventory *)

        else begin
        let slot = short_of_str_off buff (islot_off + itype) in
        if slot <> 65535 then begin
        write_empty_inv_slot buff (islot_off + 42) slot;
        write_short buff (islot_off + itype) num_items
        end else
        write_short buff (islot_off + itype) num_items
        end;

        (* Check if it's a weapon and select the appropriate slot *)

        let regex = Str.regexp "WEAPON[1-4]" in
        if i.equip &&
        Str.string_match regex i.item_slot 0 then begin
        let i_num = String.sub i.item_slot 6 1 in
        let i_num = int_of_string i_num in
        write_short buff (islot_off + 76) (i_num - 1)
        end;

        (* Splice it all together *)

        let splice_off = items_off + (num_items * 20) in
        let before_buff = Str.string_before buff splice_off in
        let after_buff = Str.string_after buff splice_off in
        before_buff ^ item_buff ^ after_buff

     | _ -> failwith ("ERROR: ADD_CRE_ITEM: Unknown cre version: " ^ cre_v)) ;



  | TP_CompileBAFtoBCS ->
        (try 
        let bcs = handle_script_buffer (patch_filename ^ ".BAF") buff in 
        let out_buff = Buffer.create 40960 in 
        Bcs.save_bcs game (Bcs.Save_BCS_Buffer(out_buff)) bcs ;
        Buffer.contents out_buff
         with _ -> buff)

  | TP_CompileDLGtoD ->
        handle_dlg_buffer game patch_filename buff 

  | TP_CompileDtoDLG ->
        handle_d_buffer game patch_filename buff 

  | TP_CompileBCStoBAF -> 
        (try 
        let bcs = handle_script_buffer (patch_filename ^ ".BCS") buff in 
        let out_buff = Buffer.create 40960 in 
        Bcs.print_script_text game (Bcs.Save_BCS_Buffer(out_buff)) 
          (Bcs.BCS_Print_Script(bcs)) false None ;
        Buffer.contents out_buff
         with _ -> buff) 
  

  | TP_Add_S_Item(store_args,item,charge1,charge2,charge3,flags,stock) ->

      (* Create the new item *)

      let item_buff = String.make 28 '\000' in
      String.blit item 0 item_buff 0 (String.length item);
      write_short item_buff 0xa charge1;
      write_short item_buff 0xc charge2;
      write_short item_buff 0xe charge3;

      let new_flags =
      (match (String.uppercase flags) with
      | "IDENTIFIED" -> 1
      | "UNSTEALABLE" -> 2
      | "STOLEN" -> 4
      | "IDENTIFIED&STOLEN" -> 5
      | "IDENTIFIED&UNSTEALABLE" -> 3
      | _ -> log_and_print "WARNING: ADD_STORE_ITEM: Unknown flag %s.  Defaulting to 0 for flags.\n" (String.uppercase flags);
             0)
      in

      write_int item_buff 0x10 new_flags;
      write_int item_buff 0x14 stock;

      (* Grab the state of the + and uppercase the item string *)

      let possible_plus = store_args.overwrite_store_item in
      let item = String.uppercase item in

      (* Read in the offsets that we need to update *)

      let drinksoffset = int_of_str_off buff 0x4c in
      let ipurchasedoffset = int_of_str_off buff 0x2c in
      let isaleoffset = int_of_str_off buff 0x34 in
      let cureoffset = int_of_str_off buff 0x70 in

      (* Read in the # of items for sale *)

      let numisale = int_of_str_off buff 0x38 in


      (* Grab all items *)

      let items_list = ref [] in
      let items_string = String.sub buff isaleoffset (ipurchasedoffset - isaleoffset) in
      for i = 0 to numisale - 1 do
      let item_check = String.sub items_string (i * 28) 8 in
      let item_check = get_string_of_size item_check 0 8 in
      items_list := item_check :: !items_list
      done;

      items_list := List.rev !items_list;

      (* If we have a + sign and the item already exists *)

      if not possible_plus && (List.mem item !items_list) then begin
      let i = ref 0 in
      let str_before = ref "" in
      let str_after = ref "" in
      let new_buff = ref "" in
      List.iter (fun patch_it ->

      if patch_it = item then begin

      (* Overwrite the item *)

      str_before := Str.string_before buff (isaleoffset + (!i * 28));
      str_after := Str.string_after buff (isaleoffset + (!i * 28) + 28);
      new_buff := !str_before ^ item_buff ^ !str_after;
      end
      else
      incr i
      ) !items_list;
      !new_buff
      end

      (* No + sign and the item exists *)

      else if possible_plus && (List.mem item !items_list) then begin
      log_and_print "%s.ITM is already in the store.  Skipping...\n" item;
      buff
      end

      (* No + sign and the item doesn't exist *)

      else begin

      log_and_print "Patching %s.ITM into store...\n" (String.uppercase item);

      (* Update the offsets by 28 bytes *)

      write_int buff 0x2c (ipurchasedoffset + 28);
      write_int buff 0x70 (cureoffset + 28);

      (* Add 1 to the #items for sale *)

      write_int buff 0x38 (numisale + 1);

      (* Splice in the new item *)

      let before = Str.string_before buff isaleoffset in
      let after = Str.string_after buff isaleoffset in
      let buff = before ^ item_buff ^ after in
      buff
      end

  | _ -> buff 
in (* end: process_patch2 *) 

  handle_recursive_tp2 := enqueue_tp2_filename ; 

  let old_allow_missing = !Load.allow_missing in 
  Load.allow_missing := 
    List.fold_left (fun acc elt -> match elt with
      Allow_Missing(lst) -> lst @ acc
      | _ -> acc) [] tp.flags ; 

  let old_script_style = game.Load.script_style in
  List.iter (fun f -> match f with
  | Script_Style(s) -> game.Load.script_style <- s
  | _ -> ()
  ) tp.flags ; 

  let interactive = ref true in 
  let our_lang = ref None in
  let our_lang_index = ref 0 in 
  (* pick your language *)
  begin 
    match tp.languages with
      [] -> ()
    | [l] -> our_lang := Some(l) 
    | _ -> 
      let arr = Array.of_list tp.languages in
      let answer, answer_index = 
        match !always_uninstall, (installed_lang_index this_tp2_filename) with
        | true, Some(i) when i >= 0 && (i < Array.length arr) -> 
            Some(arr.(i)), i 
        | _, _ -> None, 0
      in
      let answer, answer_index = ref answer, ref answer_index in 
      while !answer = None do  
        log_and_print "\nChoose your language:\n" ;
        Array.iteri (fun i l -> 
          log_and_print "%2d [%s]\n" i l.lang_name) arr ;
        try 
          let i = read_int () in
          if i >= 0 && i < Array.length arr then begin
            answer := Some(arr.(i)) ;
            answer_index := i 
          end 
        with _ -> () 
      done ;
      our_lang := !answer ;
      our_lang_index := !answer_index 
  end ;
      (* 2da helper functions *)
      let split_apart str = Str.split many_whitespace_regexp  str 
      in 
      let get_next_col_number file =
        let (a,b) = split file in 
        let buff,path = Load.load_resource "getting 2DA columnns" game true a b in 
        try 
          let lst = Str.split many_newline_or_cr_regexp buff in
          let elt = List.nth lst 5 in
          let cols = split_apart elt in 
          let num_cols = List.length cols in 
          let last_col = num_cols - 2 in 
          last_col + 1
        with e -> 
          log_and_print "ERROR: cannot find col numbers in %s\n" file ;
          raise e
      in 
      let get_next_line_number file =
        let (a,b) = split file in 
        let buff,path = Load.load_resource "getting 2DA lines" game true a b in 
        try 
          let idx = Str.search_backward (Str.regexp "[\r\n][0-9]") buff 
            ((String.length buff) - 1) in 
          let minibuff = Str.string_after buff (idx+1) in
          let lst = split_apart minibuff in
          let elt = List.hd lst in
          let last_number = int_of_string elt in
          last_number + 1
        with e -> 
          log_and_print "ERROR: cannot find line numbers in %s\n" file  ;
          raise e
      in 
      (* 2da helper functions end *)
  let rec process_action tp a = 
    try 
    (match a with
    | TP_Require_File(file,error_msg) ->
        log_and_print "Checking for required files ...\n" ;
        let size = file_size file in 
        if size >= 0 then begin
          log_or_print "[%s] %d bytes" file size 
        end else begin
          log_and_print "[%s] not found\n" file ;
          log_and_print "\n%s\n" (Dc.single_string_of_tlk_string game
          error_msg) ;
          failwith file
        end

    | TP_Inlined_File(name,contents) ->
        log_only "Defined Inlined File [%s] (length %d)\n" 
          name (String.length contents) ;
        Hashtbl.add inlined_files name contents 

    | TP_Forbid_File(file,error_msg) ->
        log_and_print "Checking for forbidden files ...\n" ;
        let size = file_size file in 
        if size >= 0 then begin
          log_and_print "[%s] found: %d bytes\n" file size ;
          log_and_print "\n%s\n" (Dc.single_string_of_tlk_string game
          error_msg) ;
          failwith file
        end else begin
          log_or_print "[%s] not found (as desired)\n" file ;
        end

    | TP_Print(msg) -> 
        let str = Dc.single_string_of_tlk_string game msg in 
        let str = Var.substitute str in 
        log_and_print "\n%s\n" str 

    | TP_Fail(msg) -> 
        let str = Dc.single_string_of_tlk_string game msg in 
        log_and_print "FAILURE:\n%s\n" str ;
        failwith str 

    | TP_If(p,al1,al2) -> 
        let res = eval game p in
        log_or_print "IF evaluates to %b\n" res ;
        if res then begin
          List.iter (process_action tp) al1
        end else begin
          List.iter (process_action tp) al2
        end 

    | TP_Uninstall_Now(name,comp) -> 
        if already_installed name comp then begin
          if uninstall game handle_tp2_filename name comp false then
            () 
          else 
            failwith 
              (Printf.sprintf "unable to uninstall %s component #%d"
                name comp) 
        end else
          log_or_print "%s component #%d not present, good.\n"
            name comp 

    | TP_RandomSeed(i) -> Random.init i

    | TP_CopyRandom(slist,plist) ->
        log_and_print "Randomizing %d file(s) ...\n" (List.length slist) ; 
        let blist = List.map (fun s -> 
          let a,b = split s in 
          let buff,path = Load.load_resource "COPY_RANDOM" game true a b in
          (s,buff)
        ) slist in
        let dlist = List.map (fun s -> "override/" ^ s) slist in
        let rand_dlist = List.sort 
          (fun a b -> if Random.bool () then 1 else -1) dlist in
        List.iter2 (fun (s,buff) d ->
          List.iter (fun p -> process_patch1 s game buff p) plist ; 
          let result_buff = 
            List.fold_left (fun acc elt -> 
              try process_patch2 s game acc elt
              with e -> 
                log_and_print "ERROR: [%s] -> [%s] Patching Failed (COPY_RANDOM) (%s)\n" 
                  s d (Printexc.to_string e); raise e) 
              buff plist 
          in 
            Stats.time "saving files" (fun () -> 
            let out = open_for_writing d true in
            output_string out result_buff ;
            close_out out ;
            log_only "Copied [%s] to [%s]\n" s d) ()
        ) blist rand_dlist 

    | TP_Copy(copy_args) -> 
        let get_existing = copy_args.copy_get_existing in
        let use_reg = copy_args.copy_use_regexp in
        let use_glob = copy_args.copy_use_glob in 
        let slist = copy_args.copy_file_list in
        let plist = copy_args.copy_patch_list in 
        let clist = copy_args.copy_constraint_list in
        let make_a_backup = copy_args.copy_backup in 

        let slist = 
          if get_existing = true && use_reg = true then begin
            let files_in_chitin = Key.list_of_key_resources game.Load.key in 
            let files_in_override = 
              if use_glob then begin
                try 
                  let dh = Unix.opendir "override" in
                  let lst = ref [] in 
                  (try 
                    while true do 
                      let next = Unix.readdir dh in
                      if ((Unix.stat ("override/" ^ next)).Unix.st_kind = 
                         Unix.S_REG) then 
                        lst := (String.uppercase next) :: !lst
                    done 
                  with End_of_file -> () );
                  !lst
                with _ -> [] 
              end else [] 
            in 
            let new_list = List.map (fun (s,p) ->
              let regexp = Str.regexp_case_fold s in 
              let matches = ref [] in 
              List.iter (fun possible ->
                if Str.string_match regexp possible 0 then begin
                  matches := (possible, p ^ "/" ^ possible) :: !matches
                end 
              ) (files_in_chitin @ files_in_override);
              let matches = List.sort compare !matches in 
              let matches = 
                let rec nodup lst = match lst with
                | [] -> []
                | [hd] -> lst 
                | a :: b :: tl -> if (a = b) then nodup (b :: tl)
                                             else a :: (nodup (b :: tl)) 
                in nodup matches 
              in 
              if (matches = []) then
                [(s,p)]
              else
                matches
            ) slist in 
            List.flatten new_list 
          end else if use_glob then begin
            let res = ref [] in 
            List.iter (fun (s,p) -> 
              let myfun (glob_s : string) = begin
                log_only "Callback from Arch.glob: %s\n" glob_s; 
                res := (glob_s, p) :: !res
              end in
              log_only "Calling Arch.glob: %s\n" s; 
              let res = (Arch.glob s myfun) in 
              log_only "Arch.glob returned.\n"; 
              res
            ) slist ;
            !res
          end else 
            slist
        in 

        let len = List.length slist in 
        log_and_print "Copying%s %d file%s ...\n" 
          (if List.length plist > 0 then " and patching" else "")
          len
          (if len = 1 then "" else "s") ;

        let copy_one_file src dest = begin
          let src_dir = Filename.dirname src in
          Var.add_var "SOURCE_DIRECTORY" src_dir ; 
          Var.add_var "SOURCE_FILESPEC" src ; 
          Var.add_var "SOURCE_FILE" (Filename.basename src) ; 
          Var.add_var "SOURCE_RES" 
            (let a,b = split (Filename.basename src) in a) ;
          let dest_dir = Filename.dirname dest in 
          Var.add_var "DEST_DIRECTORY" dest_dir ; 
          Var.add_var "DEST_FILESPEC" dest ; 
          Var.add_var "DEST_FILE" (Filename.basename dest) ; 
          Var.add_var "DEST_RES" 
            (let a,b = split (Filename.basename dest) in a) ; 
          let dest = Var.substitute_user_vars_only dest in 
          let buff = 
            if not get_existing then 
              load_file src 
            else
              let a,b = split src in 
              let buff,path = Load.load_resource "COPY" game true a b in
              buff
          in
          Var.add_var "SOURCE_SIZE" (Printf.sprintf "%d" (String.length buff)); 
          let orig_buff = 
            if List.mem TP_ButOnlyIfItChanges clist then 
              String.copy buff
            else
              ""
          in 

          (* if (buff <> "") then *) begin 
            List.iter (fun p -> process_patch1 src game buff p) plist ; 
            let ok_to_copy = List.fold_left (fun acc elt -> acc && 
              match elt with
              | TP_Contains(s) -> begin
                  let my_regexp = Str.regexp_case_fold s in
                  try let _ = Str.search_forward my_regexp buff 0 in
                    true
                  with _ -> 
                    log_only "Not copying [%s] to [%s] because it does NOT contain [%s]\n" src dest s ; 
                    false
                  end
              | TP_NotContains(s) -> begin
                  let my_regexp = Str.regexp_case_fold s in
                  try let _ = Str.search_forward my_regexp buff 0 in
                    log_only "Not copying [%s] to [%s] because it DOES contain [%s]\n" src dest s ; 
                    false
                  with _ -> 
                    true
                  end
              | TP_IfSizeIs(size) -> 
                  if String.length buff = size then 
                    true
                  else begin
                    log_only "Not copying [%s] to [%s] because size is %d, NOT %d\n" src dest (String.length buff) size ; 
                    false
                  end
              | TP_Eval(pe) -> 
                  let v = eval_pe pe in 
                  if v = Int32.zero then begin
                    log_only "Not copying [%s] to [%s] because condition evaluates to %ld\n" src dest v ; false
                  end else true 
              | TP_ButOnlyIfItChanges -> true 
            ) true clist in 
            if ok_to_copy then begin 
              let result_buff = 
                List.fold_left (fun acc elt -> 
                  try process_patch2 src game acc elt
                  with e -> log_and_print 
                    "ERROR: [%s] -> [%s] Patching Failed (COPY) (%s)\n" 
                    src dest (Printexc.to_string e); raise e) 
                buff plist 
              in 
              let dest = 
                if is_directory dest then
                  dest ^ "/" ^ (Filename.basename src)
                else 
                  dest
              in 
              let it_changed = 
                if List.mem TP_ButOnlyIfItChanges clist then 
                  begin 
                    let changed = 
                      ((String.length result_buff) <> 
                       (String.length orig_buff)) ||
                      ((String.compare orig_buff result_buff) <> 0) 
                    in
                    changed
                  end else true (* always copy *) 
              in 
              if (it_changed) then begin 
                Stats.time "saving files" (fun () -> 
                  let out = 
                    try open_for_writing_internal make_a_backup dest true 
                    with e -> log_and_print 
                      "ERROR: COPY ~%s~ ~%s~ FAILED: cannot open target\n" 
                        src dest ; 
                      raise e
                  in
                  output_string out result_buff ;
                  close_out out ;
                if make_a_backup then 
                  log_only "Copied [%s] to [%s]\n" src dest
                else 
                  log_only "Copied [%s] to [%s] (NO BACKUP MADE!)\n" src dest
                ) () (* Stats.time "saving files" *)
              end else begin
                log_only 
                  "Not copying [%s] to [%s] because it did not change\n" 
                  src dest  
              end 
            end (* end: if ok_to_copy *) 
          end (* (* end: if buff <> "" *) 
          else log_or_print "Not copying [%s] to [%s] because it is empty\n"
            src dest 
            *)
        end (* end: let copy_one_file = begin *)  
        in 
        List.iter (fun (src,dest) -> 
          if is_directory src then begin
            try
              let dh = Unix.opendir src in 
              while true do
                let base = Unix.readdir dh in 
                let source = src ^ "/" ^ base in 
                if not (is_directory source) then
                  copy_one_file source (dest ^ "/" ^ base)
              done 
            with _ -> () 
          end else copy_one_file src dest 
        ) slist 

    | TP_Add_Music(m) -> begin
      log_and_print "Adding %s Music ...\n" m.music_name; 
      let this_music_number = get_next_line_number "SONGLIST.2DA" in 
      let music_base_name = Filename.basename m.music_file in 
      let str_to_append = Printf.sprintf "%d %s %s"
        this_music_number m.music_name music_base_name in 

      let a1 = TP_Append("SONGLIST.2DA",str_to_append,[]) in

      let dest_music_file = "music/" ^ music_base_name in 
      let a2 = TP_Copy(
        { copy_get_existing = false;
          copy_use_regexp = false;
          copy_use_glob = false;
          copy_file_list = [(m.music_file,dest_music_file)] ;
          copy_patch_list = [] ;
          copy_constraint_list = [] ;
          copy_backup = true; } ) in 
      let action_list = [ a1 ; a2 ] in 
      List.iter (process_action tp) action_list ; 
      Var.add_var (m.music_name) (string_of_int this_music_number) ;
      log_and_print "Added %s Music\n" m.music_name; 
    end

    | TP_Add_Projectile(p) -> begin
        log_and_print "Adding projectile file %s ...\n" p.pro_file;
        let this_pro_name = Filename.chop_extension (Filename.basename p.pro_file) in 
        let this_pro_number = get_next_line_number "PROJECTL.IDS" in
        let a1 = TP_Append("PROJECTL.IDS", 
                           (Printf.sprintf "%d %s" this_pro_number this_pro_name),[]) in
        let dest_pro_file = "override/" ^ (Filename.basename p.pro_file) in
        let a2 = TP_Copy(
          { copy_get_existing = false;
            copy_use_regexp = false;
            copy_use_glob = false;
            copy_file_list = [(p.pro_file, dest_pro_file)] ;
            copy_patch_list = [] ;
            copy_constraint_list = [] ;
            copy_backup = true; } ) in 
        let action_list = [ a1 ; a2 ] in 
        List.iter (process_action tp) action_list ; 
        Var.add_var (this_pro_name) (string_of_int (this_pro_number + 1)) ;
        log_and_print "Added projectile file %s\n" p.pro_file; 
    end

    | TP_Add_Kit(k) -> begin
      log_and_print "Adding %s Kit ...\n" k.kit_name; 

      if eval game 
        (Pred_And(
        (Pred_File_Exists("override/kitlist.2da")),
        (Pred_File_Contains("override/kitlist.2da",k.kit_name)))) then begin
        log_and_print "\n\nERROR: Kit [%s] already present! Skipping!\n\n"
        k.kit_name 
      end else begin 
        let a1 = TP_Append("CLASWEAP.2DA",k.clasweap,[]) in
        let a2 = TP_Append_Col("WEAPPROF.2DA",
          ("" :: "" :: (split_apart k.weapprof)),[]) in
        let a3 = TP_Append("ABCLASRQ.2DA",k.abclasrq,[]) in
        let a4 = TP_Append("ABDCDSRQ.2DA",k.abdcdsrq,[]) in
        let a5 = TP_Append("ABDCSCRQ.2DA",k.abdcscrq,[]) in
        let a_e1 = TP_Append("ABCLSMOD.2DA",k.abclsmod,[]) in
        let a_e2 = TP_Append("DUALCLAS.2DA",k.dualclas,[]) in
        let a6 = TP_Append("ALIGNMNT.2DA",k.alignmnt,[]) in
        let abil_file = String.uppercase (Filename.basename k.ability_file) in 
        let abil_file_no_ext = Filename.chop_extension abil_file in 
        let dest_abil_file = "override/" ^ abil_file in 
        let a7 = TP_Copy(
          { copy_get_existing = false ;
            copy_use_regexp = false;
            copy_use_glob = false;
            copy_file_list = [(k.ability_file,dest_abil_file)] ;
            copy_patch_list = [] ;
            copy_constraint_list = [] ;
            copy_backup = true ; } ) in 
        let include_list = split_apart k.include_in in
        let lower_index = match Dc.resolve_tlk_string game k.lower with
          Dlg.TLK_Index(i) -> i
        | _ -> log_and_print "ERROR: cannot resolve KIT lower string\n" ; 
                failwith "resolve" 
        in 
        let mixed_index = match Dc.resolve_tlk_string game k.mixed with
          Dlg.TLK_Index(i) -> i
        | _ -> log_and_print "ERROR: cannot resolve KIT mixed string\n" ; 
                failwith "resolve" 
        in 
        let help_index = match Dc.resolve_tlk_string game k.help with
          Dlg.TLK_Index(i) -> i
        | _ -> log_and_print "ERROR: cannot resolve KIT help string\n" ; 
                failwith "resolve" 
        in 
        let this_kit_number = get_next_line_number "KITLIST.2DA" in
        let this_kit_prof_number = get_next_col_number "WEAPPROF.2DA" in 
        let append_to_kitlist = Printf.sprintf 
          "%d  %s %d %d %d %s %d %s" 
          this_kit_number k.kit_name 
          lower_index mixed_index help_index
          abil_file_no_ext this_kit_prof_number 
          k.unused_class in
        let a8 = TP_Append("KITLIST.2DA",append_to_kitlist,[]) in
        let include_actions = List.map (fun file -> 
          let num = get_next_line_number (file ^ ".2DA" ) in
          let str = Printf.sprintf "%d  %d" num this_kit_number in
          TP_Append(file ^ ".2DA",str,[]) 
        ) include_list in 
        let abbr = Printf.sprintf  "%s     %s" k.kit_name k.tob_abbr in
        let a9 = TP_Append("LUABBR.2DA",abbr,[]) in 
        let a10 = TP_Set_Col("25STWEAP.2DA",
          ("" :: "" :: k.kit_name :: k.tob_start),this_kit_prof_number+1) in 
        let a11 = TP_Append("KIT.IDS",
          (Printf.sprintf "0x%x %s" (0x4000 + this_kit_number)
            k.kit_name),[]) in 
        let action_list = a1 :: a2 :: a3 :: a4 :: a5 :: a6 :: a7 :: a8 :: 
          a9 :: a10 :: a11 :: a_e1 :: a_e2 :: include_actions in
        let old_allow_missing = !Load.allow_missing in 
        Load.allow_missing := 
          "LUABBR.2DA" :: "25STWEAP.2DA" :: old_allow_missing ; 
        (* actually do it! *)
        List.iter (process_action tp) action_list ; 
        Load.allow_missing := old_allow_missing ; 
        Var.add_var (k.kit_name) (string_of_int this_kit_number) ;
        log_and_print "Added %s Kit\n" k.kit_name; 
        Bcs.clear_ids_map game ; 
      end
    end 

    | TP_String_Set(s1,str,tra_file_opt) -> begin
      let i_list = try
        [int_of_string s1]
      with 
        _ -> begin
              try 
                Hashtbl.find_all game.Load.dialog_search s1 
              with _ ->
                log_and_print "ERROR: Cannot find string [%s]\n" s1 ;
                failwith "ERROR: STRING_SET"
             end 
      in 
      (match tra_file_opt with 
      | None -> ()
      | Some(tra_file) -> 
        begin 
        Dc.push_copy_trans ();
        handle_tra_filename tra_file
        end
      ) ; 
      List.iter (fun i -> 
        Dc.set_string game i str false 
      ) i_list ;
      (match tra_file_opt with 
      | None -> ()
      | Some(_) -> Dc.pop_trans ())
      end

    | TP_Mkdir(str_l) -> begin
      log_and_print "Creating %d directories\n" (List.length str_l) ;
      List.iter (fun str ->
        try 
          Unix.mkdir str 511 (* 511 = octal 0777 = a+rwx *)
        with _ -> 
          () 
      ) str_l 
      end

    | TP_Compile(dlg_l,tra_l) -> begin
        let numd = ref 0 in
        let nums = ref 0 in 
        let handle_one_d_file filespec = match split 
          (String.uppercase filespec) with
        | _,"BAF" -> incr nums 
        | _,"D" -> incr numd 
        | _,_ -> () 
        in 
        List.iter (fun filespec -> 
          if is_directory filespec then begin
            try
              let dh = Unix.opendir filespec in 
              while true do
                let base = Unix.readdir dh in 
                let source = filespec ^ "/" ^ base in 
                if not (is_directory source) then
                  handle_one_d_file source 
              done 
            with _ -> () 
          end else handle_one_d_file filespec 
        ) dlg_l ; 
        log_and_print "Compiling " ;
        (if (!numd > 0) then begin
          log_and_print "%d dialogue file%s " !numd
            (if (!numd > 1) then "s" else "") ;
          if (!nums > 0) then
            log_and_print "and "
          end
        ) ;
        (if (!nums > 0) then begin
          log_and_print "%d script%s " !nums
            (if (!nums > 1) then "s" else "") ;
          end
        ) ; 
        log_and_print "...\n" ; 

        Dc.push_copy_trans ();
        begin 
          match !our_lang with
            Some(l) -> List.iter (fun path -> 
                let my_regexp = Str.regexp_string "%s" in
                let tra_file = Str.global_replace 
                  my_regexp l.lang_dir_name path in
                handle_tra_filename tra_file ;
              ) tra_l 
          | _ -> List.iter (fun tra_file -> 
                handle_tra_filename tra_file ; 
              ) tra_l 
        end ;
        let handle_one_d_file d = 
          (* handle AUTO_TRA "solarom/%s" *)
          Dc.push_copy_trans () ;
          List.iter (fun f -> 
            match f,!our_lang with
            Auto_Tra(path),Some(l) -> 
                  let my_regexp = Str.regexp_string "%s" in
                  let tra_file_dir = Str.global_replace 
                    my_regexp l.lang_dir_name path in
                  let d_base,_ = split (Filename.basename d) in
                  let tra_file = tra_file_dir ^ "/" ^ d_base ^ ".TRA" in 
                  handle_tra_filename tra_file ;
          | Auto_Tra(path),None -> 
                  let d_base,_ = split (Filename.basename d) in
                  let tra_file = path ^ "/" ^ d_base ^ ".TRA" in 
                  handle_tra_filename tra_file 
          | _ -> () 
          ) tp.flags ;
          Dc.ok_to_resolve_strings_while_loading := Some(game) ; 
          (try 
            begin 
            match split (String.uppercase (Filename.basename d)) with
            | _,"BAF" -> compile_baf_filename d
            | _,"D" -> handle_d_filename d 
            | _,_ -> () 
            end 
          with e -> 
            begin 
              Dc.ok_to_resolve_strings_while_loading := None ; 
              Dc.pop_trans () ; 
              log_and_print "ERROR: compiling [%s]!\n" d ;
              raise e
            end
          ) ;
          Dc.ok_to_resolve_strings_while_loading := None ; 
          Dc.pop_trans () ; 
        in 
        List.iter (fun filespec -> 
          if is_directory filespec then begin
            try
              let dh = Unix.opendir filespec in 
              while true do
                let base = Unix.readdir dh in 
                let source = filespec ^ "/" ^ base in 
                if not (is_directory source) then
                  handle_one_d_file source 
              done 
            with End_of_file -> () 
          end else handle_one_d_file filespec 
        ) dlg_l ; 
        log_or_print "Processing %d dialogues/scripts ...\n" 
          (List.length dlg_l) ; 
        emit_dlg_files game "override" ;
        Dc.pop_trans () ;
        end

    | TP_Set_Col(file,new_col_list,col_num) -> 
        log_and_print "Setting game text column-wise ...\n" ; 
        let eight,three = split (String.uppercase file) in 
        let buff,loaded_path = Load.load_resource "SET_COLUMN" game true eight three in
        if buff = "" then 
          log_or_print "[%s]: empty or does not exist\n" file
        else begin 
          let dest = "override/" ^ file in 
          let buff_as_lines = Str.split many_newline_or_cr_regexp buff in
          if List.length buff_as_lines <> List.length new_col_list then begin
            log_and_print "Cannot set column-wise because there are %d lines in %s but I was only given %d things to append\n" (List.length buff_as_lines) file (List.length new_col_list)  ;
            failwith ("cannot set column-wise to " ^ file)
          end ;
          Stats.time "saving files" (fun () -> 
          let out = open_for_writing dest true in
          List.iter2 (fun orig_line new_col ->
            let line_as_cols = Str.split many_whitespace_regexp orig_line in 
            let i = ref 0 in 
            List.iter (fun orig_col -> 
              (if !i = col_num then 
                Printf.fprintf out "%-20s" new_col
              else
                Printf.fprintf out "%-20s" orig_col) ;
              incr i;
            ) line_as_cols ;
            (if (!i <= col_num) then 
                Printf.fprintf out "%-20s" new_col) ; 
            output_string out "\r\n" 
          ) buff_as_lines new_col_list ; 
          close_out out) () ; 
          log_or_print "Set text in [%s] column-wise\n" file 
        end 

    | TP_Append_Col(file,src_list,con_l) ->
        log_and_print "Appending game text column-wise ...\n" ; 
        let eight,three = split (String.uppercase file) in 
        let buff,loaded_path = Load.load_resource "APPEND_COLUMN" game true eight three in
        if buff = "" then 
          log_or_print "[%s]: empty or does not exist\n" file
        else begin 
        let okay = List.fold_left (fun acc elt -> acc &&
          match elt with 
            TP_Contains(s) -> begin
              let my_regexp = Str.regexp_case_fold s in
              try let _ = Str.search_forward my_regexp buff 0 in
                log_only "May append cols to [%s] because it DOES contain [%s]\n" file s  ; 
                true
              with _ -> 
                log_only "Not appending cols to [%s] because it does NOT contain [%s]\n" file s ; 
                false
              end
          | TP_NotContains(s) -> begin
              let my_regexp = Str.regexp_case_fold s in
              try let _ = Str.search_forward my_regexp buff 0 in 
                log_only "Not appending cols to [%s] because it DOES contains [%s]\n" file s ; 
                false
              with _ -> 
                log_only "May append cols to [%s] because it does NOT contain [%s]\n" file s ; 
                true
              end
          | TP_IfSizeIs(size) -> String.length buff = size 
          | TP_ButOnlyIfItChanges -> true 
          | TP_Eval(pe) -> 
                let v = eval_pe pe in 
                if v = Int32.zero then begin
                  log_only "Not appending cols to [%s] because condition evaluates to %ld\n" file v ; false
                end else begin 
                  log_only "May append cols to [%s] because condition evaluates to %ld\n" file v ; true
                end 
        ) true con_l in 
        if okay then begin (* do the append *) 
          let dest = "override/" ^ file in 
          let buff_as_lines = Str.split many_newline_or_cr_regexp buff in
          if List.length buff_as_lines <> List.length src_list then begin
            log_and_print "Cannot append column-wise because there are %d lines in %s but I was only given %d things to append\n" (List.length buff_as_lines) file (List.length src_list)  ;
            failwith ("cannot append column-wise to " ^ file)
          end ;
          Stats.time "saving files" (fun () -> 
          let out = open_for_writing dest true in
          List.iter2 (fun orig app ->
            output_string out orig ;
            output_string out " " ;
            output_string out app ;
            output_string out "\r\n" 
          ) buff_as_lines src_list ; 
          close_out out) () ; 
          log_or_print "Appended text to [%s] column-wise\n" file 
        end 
        end

    | TP_Append(file,src,con_l) ->
        log_and_print "Appending game text ...\n" ; 
        let eight,three = split (String.uppercase file) in 
        let buff,loaded_path = Load.load_resource "APPEND" game true eight three in
        let okay = List.fold_left (fun acc elt -> acc &&
          match elt with 
            TP_Contains(s) -> begin
              let my_regexp = Str.regexp_case_fold s in
              try let _ = Str.search_forward my_regexp buff 0 in
                log_only "May append [%.10s...] to [%s] because it DOES contain [%s]\n" src file s  ; 
                true
              with _ -> 
                log_only "Not appending [%.10s...] to [%s] because it does NOT contain [%s]\n" src file s ; 
                false
              end
          | TP_NotContains(s) -> begin
              let my_regexp = Str.regexp_case_fold s in
              try let _ = Str.search_forward my_regexp buff 0 in 
                log_only "Not appending [%.10s...] to [%s] because it DOES contains [%s]\n" src file s ; 
                false
              with _ -> 
                log_only "May append [%.10s...] to [%s] because it does NOT contain [%s]\n" src file s ; 
                true
              end
          | TP_IfSizeIs(size) -> String.length buff = size 
          | TP_ButOnlyIfItChanges -> true
          | TP_Eval(pe) -> 
                let v = eval_pe pe in 
                if v = Int32.zero then begin
                  log_only "Not appending [%.10s] to [%s] because condition evaluates to %ld\n" src file v ; false
                end else begin 
                  log_only "May appending [%.10s] to [%s] because condition evaluates to %ld\n" src file v ; true
                end 
        ) true con_l in 
        if okay then begin (* do the append *) 
          let dest = "override/" ^ file in 
          Stats.time "saving files" (fun () -> 
          let out = open_for_writing dest true in
          let nice_newlines = Str.global_replace
            many_newline_or_cr_regexp "\r\n" (buff ^ "\r\n") in
          output_string out nice_newlines ; 
          (* output_string out buff ;
          output_string out "\r\n" ; *)
          output_string out src ;
          output_string out "\r\n" ;
          close_out out) () ; 
          log_or_print "Appended text to [%s]\n" file 
        end 

    | TP_Extend_Top(use_reg,dest,src,pl,tra_l) 
    | TP_Extend_Bottom(use_reg,dest,src,pl,tra_l) -> begin
        log_and_print "Extending game scripts ...\n" ; 
        let dlist = 
          if use_reg = false then
            [dest]
          else begin
            let files_in_chitin = Key.list_of_key_resources game.Load.key in 
            let regexp = Str.regexp_case_fold dest in 
            let matches = ref [] in 
            List.iter (fun possible ->
              if Str.string_match regexp possible 0 then begin
                matches := (possible) :: !matches
              end 
            ) files_in_chitin ;
            if (!matches = []) then
              [dest]
            else
              !matches
          end 
        in 
        Dc.push_copy_trans () ;
        begin 
          match !our_lang with
            Some(l) -> List.iter (fun path -> 
                let my_regexp = Str.regexp_string "%s" in
                let tra_file = Str.global_replace 
                  my_regexp l.lang_dir_name path in
                handle_tra_filename tra_file ;
              ) tra_l 
          | _ -> List.iter (fun tra_file -> 
                handle_tra_filename tra_file ; 
              ) tra_l 
        end ;
        let src_script = 
          try 
            let src_buff = load_file src in 
            List.iter (fun p -> process_patch1 src game src_buff p) pl ; 
            let src_buff = List.fold_left (fun acc elt -> 
                  try process_patch2 src game acc elt
                  with e -> 
                    log_and_print "ERROR: [%s] -> [%s] Patching Failed (EXTEND_TOP/BOTTOM)\n" 
                      src dest ; raise e) 
                src_buff pl 
            in 
            Dc.ok_to_resolve_strings_while_loading := Some(game) ; 
            (try 
              let res = handle_script_buffer src src_buff in 
              Dc.ok_to_resolve_strings_while_loading := None ; 
              res
            with e -> 
              begin 
              Dc.ok_to_resolve_strings_while_loading := None ; 
              raise e
              end 
            ) ;
          with _ -> []
        in 
        List.iter (fun dest -> 
          let eight,three = split (String.uppercase dest) in 
          let dest_script = 
            let old_a_m = !Load.allow_missing in 
            Load.allow_missing := dest :: old_a_m ;
            let dest_buff, dest_path = 
              try 
                Load.load_resource "EXTEND_TOP/EXTEND_BOTTOM" game true eight three 
              with _ -> 
                begin 
                log_only "[%s] not found, treating as empty.\n" dest ;
                "",""
                end 
            in 
            Load.allow_missing := old_a_m ; 
            handle_script_buffer dest dest_buff 
          in 

          let destpath = "override/" ^ dest in 
          Stats.time "saving files" (fun () -> 
          let out = open_for_writing destpath true in
          Bcs.save_bcs game (Bcs.Save_BCS_OC(out)) (match a with 
            TP_Extend_Top(_,_,_,_,_) -> src_script @ dest_script
                                 | _ -> dest_script @ src_script) ;
          close_out out) () ; 
          log_or_print "Extended script [%s] with [%s]\n" dest src
        ) dlist ;
        Dc.pop_trans () ; 
        end

    | TP_At_Interactive_Exit(str) ->  
        if !interactive then process_action tp (TP_At_Exit(str))
    | TP_At_Interactive_Uninstall(str) ->
        if !interactive then process_action tp (TP_At_Uninstall(str))

    | TP_At_Exit(str) ->  
        begin 
        let str = Var.substitute_user_vars_only str in 
        let a,b = split (String.uppercase str) in 
        match b with 
        | "TP2" -> (!handle_recursive_tp2) str 
        | _ -> 
          let str = Arch.handle_view_command str in 
          if List.mem str !execute_at_exit then
            ()
          else
            execute_at_exit := str :: !execute_at_exit 
        end 

    | TP_At_Uninstall(str) -> () 
    )
    with e -> (* from: let rec process_action = try *)
      (if !continue_on_error then begin
        log_and_print "WARNING: Continuing despite [%s]\n"
          (Printexc.to_string e)
      end else begin
        log_and_print "Stopping installation because of error.\n" ; 
        raise e
      end)
  in 
  let lang_init () = 
    init_default_strings () ; 
    begin 
      match !our_lang with
        None -> () 
      | Some(l) -> 
          log_and_print "Using Language [%s]\n" l.lang_name ;
          Var.add_var "LANGUAGE" l.lang_dir_name ; 
          log_or_print "[%s] has %d top-level TRA files\n"
            l.lang_name (List.length l.lang_tra_files) ;
          List.iter handle_tra_filename l.lang_tra_files
    end 
  in 

  lang_init () ; 

  let subcomp_group the_comp = 
    let rec walk lst = match lst with
    | TPM_SubComponents(ts,_) :: tl -> Some(ts)
    | hd :: tl -> walk tl
    | [] -> None
    in walk the_comp.mod_flags 
  in 
  let subcomp_predicate the_comp = 
    let rec walk lst = match lst with
    | TPM_SubComponents(_,p) :: tl -> eval game p 
    | hd :: tl -> walk tl
    | [] -> false
    in walk the_comp.mod_flags 
  in 

  let last_module_index = get_last_module_index tp in 
  let comp_num = ref 0 in 
  let comp_ht = Hashtbl.create 255 in
  List.iter (fun the_mod ->
    incr comp_num ; 
    (match subcomp_group the_mod with
    | Some(ts) -> 
        if Hashtbl.mem comp_ht ts then
          decr comp_num
        else
          Hashtbl.add comp_ht ts true 
    | None -> ()) 
  ) tp.module_list ; 
  let comp_num = !comp_num in
  (* comp_num = number of user visible "component chunks", each of which
   * can have multiple sub-components *) 
   
  let any_member_of_subcomp_group_installed the_comp = 
    let i = ref 0 in 
    List.exists (fun the_mod ->
      match subcomp_group the_mod with
      | Some(ts) -> if ts = the_comp && 
                       already_installed this_tp2_filename !i then
                      true
                    else (incr i ; false)
      | None -> (incr i; false)
    ) tp.module_list
  in 


  let any_already_installed = ref false in 
  let any_not_yet_installed = ref false in 

  let subcomp_installed = Hashtbl.create 255 in 

  for i = 0 to last_module_index do 
    try let m = get_nth_module tp i in
    if already_installed this_tp2_filename i then begin 
      any_already_installed := true ;
      match subcomp_group m with
      | Some(ts) -> Hashtbl.add subcomp_installed ts true
      | None -> ()
    end else begin
      let deprecated = List.exists (fun f -> match f with 
        | TPM_Deprecated(warn) -> true
        | _ -> false) m.mod_flags 
      in 
      if not deprecated then begin
        let group_already = 
          match subcomp_group m with
          | Some(ts) -> Hashtbl.mem subcomp_installed ts
          | None -> false
        in 
        if not group_already then 
          any_not_yet_installed := true 
      end 
    end 
    with Not_found -> () 
  done ; 

  let module_defaults = Array.init (last_module_index+1) (fun i -> 
      if !always_yes then
        TP_Install
      else if !always_uninstall && (already_installed this_tp2_filename i) then
        TP_Uninstall
      else if !always_uninstall then
        TP_Skip
      else 
        TP_Ask) 
  in

  let get_trans i = Dc.single_string_of_tlk_string game (Dlg.Trans_String(i)) in

  let handle_letter tp answer can_uninstall package_name m finished i =
    let subgroup_already = 
      match subcomp_group m with
      | Some(ts) -> any_member_of_subcomp_group_installed ts
      | None -> false
    in 

    match answer with 
    | "Q" -> begin
        for i = 0 to last_module_index do
          module_defaults.(i) <- TP_Skip ; 
          finished := true 
        done 
        end 
    | "Y" when subgroup_already -> 
        log_or_print "Skipping [%s] because another subcomponent of [%s] is already installed.\n" package_name
        (match subcomp_group m with 
          | Some(ts) -> Dc.single_string_of_tlk_string game  ts
          | None -> "???") ;
        finished := true 

    | "Y" | "R" -> begin
        if can_uninstall then begin 
          try 
            (* log_and_print 
            "\nRemoving old installation of [%s] first ...\n" *)
            log_and_print "\n%s%s%s\n"
              ((get_trans (-1013))) package_name 
              ((get_trans (-1014))) ;
            (if not (uninstall game handle_tp2_filename this_tp2_filename i !interactive) then failwith "uninstallation error"); 
            log_and_print 
              (* "\nSUCCESSFULLY REMOVED OLD [%s]\n\n"  *)
              "\n%s [%s]\n\n" 
              ((get_trans (-1015))) package_name ;
          with e ->
            log_and_print "WARNING: unable to uninstall: %s\n" 
              (Printexc.to_string e)
        end ; 
        set_backup_dir tp.backup i ; 
        let strset_backup_filename = 
          Printf.sprintf "%s/%d/UNSETSTR.%d" tp.backup i i 
        in 
        log_and_print "\n%s [%s]\n" 
          (* "\nInstalling [%s]\n"  *)
              ((get_trans (-1016))) 
          package_name ;
        (try 
          List.iter (fun flag -> match flag with
            Always(al) -> List.iter (process_action tp) al
          | _ -> () 
          ) tp.flags ;
          List.iter (process_action tp) m.mod_parts ;
        with e -> begin
          log_and_print "\n%s%s%s\n" 
          (*  "\nERROR Installing [%s], rolling back to previous state\n" *)
              ((get_trans (-1017))) package_name ((get_trans (-1018))) ;
          Dc.clear_state () ; 
          record_strset_uninstall_info game strset_backup_filename ;
          (match !backup_list_chn with
            Some(chn) -> close_out chn ; backup_list_chn := None
          | None -> ()) ;
          uninstall_tp2_component game tp this_tp2_filename i false ;
          print_log () ; 
          raise e
        end ); 
        log_and_print "\n\n" ; 
        record_strset_uninstall_info game strset_backup_filename ;
        strings_to_print_at_exit := 
          (Printf.sprintf "\n%s [%s]\n" 
           (*  "\nSUCCESSFULLY INSTALLED [%s]\n"  *)
              ((get_trans (-1019))) 
            package_name) :: 
          !strings_to_print_at_exit ;
        (* add this successful install to the log! *)
        begin 
        if List.find_all (fun x -> x = TPM_NotInLog) m.mod_flags = [] then
          the_log := !the_log @ 
            [ ((String.uppercase this_tp2_filename),!our_lang_index,i,Some(package_name),Installed) ] 
        else log_and_print "NOT adding a WeiDU.log record. You cannot uninstall this.\n" 
        end ;
        finished := true 
      end
    | "N" -> 
        log_and_print "\n%s [%s]\n" 
         (* "\nSkipping [%s]\n" *)
              ((get_trans (-1020))) 
          package_name ;
        finished := true 
    | "U" -> 
        log_and_print "\n%s%s%s%d%s\n" 
         (* "\nRemoving [%s] (component #%d)\n" *)
              ((get_trans (-1021))) 
              package_name 
              ((get_trans (-1022))) 
              i
              ((get_trans (-1023))) ;
        (if not (uninstall game handle_tp2_filename this_tp2_filename i !interactive ) then failwith "uninstallation error" );
        log_and_print "\n\n%s%s%s%d%s\n" 
         (* "\n\nSUCCESSFULLY REMOVED [%s] (component #%d)\n\n" *)
              ((get_trans (-1024))) 
              package_name 
              ((get_trans (-1022))) 
              i
              ((get_trans (-1023))) ;
        finished := true 
    | _ -> () 
  in 

  let specify = ref false in 

  (* for big mods, ask about things in general first *) 
  if comp_num > 4 && not !always_yes && not !always_uninstall &&
     not (List.exists (fun a -> a = Ask_Every_Component) tp.flags)
  then begin
  (* add (-1000) "\nThis mod has %d distinct optional components.\nTo save time, you can choose what to do with them at a high level rather\nthan being asked about each one.\n" ; *)
      log_and_print "\n%s %d %s" (get_trans (-1000)) comp_num (get_trans (-1001)) ;
      let finished = ref false in

      if !any_not_yet_installed then 
      while not !finished do 
        finished := true ;
        (* log_and_print "\nWhat should be done with all components that are NOT YET installed?\n[I]nstall them, [S]kip them, [A]sk about each one? " ; *)
        log_and_print "\n%s" ((get_trans (-1002))); 
        match String.uppercase(read_line ()) with
        | "R"
        | "I" -> 
          for i = 0 to last_module_index do 
            try 
            let the_comp = get_nth_module tp i in
            match subcomp_group the_comp with
            | Some(x) ->
              if not (any_member_of_subcomp_group_installed x) then 
                module_defaults.(i) <- TP_Install
            | None -> 
              if not (already_installed this_tp2_filename i) then
                module_defaults.(i) <- TP_Install
            with Not_found -> () 
          done 
        | "S" ->
          for i = 0 to last_module_index do 
            try
            let the_comp = get_nth_module tp i in
            match subcomp_group the_comp with
            | Some(x) ->
              if not (any_member_of_subcomp_group_installed x) then 
                module_defaults.(i) <- TP_Skip
            | None -> 
              if not (already_installed this_tp2_filename i) then
                module_defaults.(i) <- TP_Skip
            with Not_found -> () 
          done 
        | "A" -> ()
        | "X" -> specify := true 
        | _ -> finished := false 
      done ;

      finished := false ; 
      if !any_already_installed then 
      while not !finished do
        finished := true ;
        (* log_and_print "\nWhat should be done with all components that are ALREADY installed?\n[R]e-install them, [U]ninstall them, [S]kip them, [A]sk about each one? " ; *)
        log_and_print "\n%s" ((get_trans (-1003))); 
        match String.uppercase(read_line ()) with
        | "I" 
        | "R" -> 
          for i = 0 to last_module_index do 
            try 
            let the_comp = get_nth_module tp i in
            match subcomp_group the_comp with
            | Some(x) ->
              if (any_member_of_subcomp_group_installed x) then 
                module_defaults.(i) <- TP_Install
            | None -> 
              if (already_installed this_tp2_filename i) then
                module_defaults.(i) <- TP_Install
            with Not_found -> () 
          done 
        | "S" ->
          for i = 0 to last_module_index do 
            try let the_comp = get_nth_module tp i in 
            match subcomp_group the_comp with
            | Some(x) ->
              if (any_member_of_subcomp_group_installed x) then 
                module_defaults.(i) <- TP_Skip
            | None -> 
              if (already_installed this_tp2_filename i) then
                module_defaults.(i) <- TP_Skip
            with Not_found -> () 
          done 
        | "U" ->
          for i = 0 to last_module_index do 
            try let the_comp = get_nth_module tp i in 
            match subcomp_group the_comp with
            | Some(x) ->
              if (any_member_of_subcomp_group_installed x) then 
                module_defaults.(i) <- TP_Uninstall
            | None -> 
              if (already_installed this_tp2_filename i) then
                module_defaults.(i) <- TP_Uninstall
            with Not_found -> () 
          done 
        | "A" -> ()
        | "X" -> specify := true 
        | _ -> finished := false 
      done ;
  end ; 

  let handle_error_generic always_yes finished package_name = (fun e ->
    log_and_print "ERROR: %s\n" (Printexc.to_string e) ;
    Dc.clear_state () ; 
    (if (!log_file <> "") then 
      log_and_print "%s %s %s %s\n" ((get_trans (-1004))) !log_file 
                                    (get_trans (-1005)) tp.author) ;
    (* log_and_print "PLEASE email the file %s to %s\n" !log_file tp.author);*)
    if !always_yes then begin
      log_and_print "Automatically Skipping [%s] because of error.\n"
        package_name ; 
      finished := true 
    end ; 
    lang_init ()) 
  in 

  let asked_about_comp = Hashtbl.create 255 in 

  let ask_about_module_with_subcomp current m subcomp handle_error = begin
    let subcomp_group_str = Dc.single_string_of_tlk_string game subcomp in 
    if Hashtbl.mem asked_about_comp subcomp then
      log_or_print "Already Asked About [%s]\n" subcomp_group_str 
    else begin 
      Hashtbl.add asked_about_comp subcomp true ;
    let any_already = any_member_of_subcomp_group_installed subcomp in 
    let finished = ref false in 
    while not !finished do
      try 

        (if (any_already) then 
          log_and_print "\n%s%s%s\n" (get_trans (-1006)) subcomp_group_str
            (get_trans (-1025)) 
        else 
          log_and_print "\n%s%s%s\n" (get_trans (-1006)) subcomp_group_str
            (get_trans (-1026))) ;
        let choice_num = ref 1 in
        let choice_ht = Hashtbl.create 255 in 
        let already_ht = Hashtbl.create 255 in 
        for i = 0 to last_module_index do
          try let m = get_nth_module tp i in
          match subcomp_group m with
          | Some(ts) when ts = subcomp && (subcomp_predicate m) ->
            let this_subcomp_name = Dc.single_string_of_tlk_string 
              game m.mod_name in 
            log_and_print "%2d] %s" !choice_num this_subcomp_name ; 
            (if already_installed this_tp2_filename i then begin 
              log_and_print "%s\n" (get_trans (-1027)) ;
              Hashtbl.add already_ht () (m,i) ;
              Hashtbl.add choice_ht !choice_num (m,i,true)
            end else log_and_print "\n" 
            );
            Hashtbl.add choice_ht !choice_num (m,i,false) ;
            incr choice_num 
          | _ -> () 
          with Not_found -> () 
        done ;
        let answer = String.uppercase (read_line ()) in 
        (match answer with
        | "U" | "R" ->
            if Hashtbl.mem already_ht () then begin
              let (m,i) = Hashtbl.find already_ht () in 
              let can_uninstall = already_installed this_tp2_filename i in 
              let package_name = 
                Dc.single_string_of_tlk_string game m.mod_name in 
              handle_letter tp answer can_uninstall 
                package_name m finished i ;
              finished := true 
            end 
        | "N" -> 
            finished := true; 
        | "Q" -> 
          for i = 0 to last_module_index do
            module_defaults.(i) <- TP_Skip ;
            finished := true
          done 
        | _ -> 
          begin
            let choice, ok = try (int_of_string answer,true) 
                             with _ -> (0,false)
            in
            if (ok) then 
              if Hashtbl.mem choice_ht choice then begin
                let (m,i,already) = Hashtbl.find choice_ht choice in 
                if (already) then
                  finished := true
                else begin
                  log_or_print "Uninstalling All Other Subcomponents of [%s]\n"
                    subcomp_group_str ;
                  Hashtbl.iter (fun () (m,i) -> 
                    let can_uninstall = already_installed 
                      this_tp2_filename i in 
                    let package_name = 
                      Dc.single_string_of_tlk_string game m.mod_name in 
                    handle_letter tp "U" can_uninstall 
                      package_name m finished i
                  ) already_ht ;
                  log_or_print "Done Uninstalling All Other Subcomponents of [%s]\n" subcomp_group_str ;
                  let can_uninstall = already_installed 
                    this_tp2_filename i in 
                  let package_name = 
                    Dc.single_string_of_tlk_string game m.mod_name in 
                  handle_letter tp "Y" can_uninstall 
                    package_name m finished i ;
                  finished := true
                end 
              end 
          end 
        )
      with e -> handle_error e

    done 
    end 
  end in

  let ask_about_module current m package_name handle_error = begin
    let finished = ref false in 
        while not !finished do try
          let can_uninstall = already_installed this_tp2_filename !current in 
          if can_uninstall then 
            (* log_and_print "\nInstall Component [%s]\n[R]e-Install, [N]o Change or [U]ninstall or [Q]uit? "  package_name *)
            log_and_print "\n%s%s%s" (get_trans (-1006)) package_name
              (get_trans (-1007)) 
          else 
           (* log_and_print "\nInstall Component [%s]\n[Y]es or [N]o or [Q]uit? "  package_name ; *)
            log_and_print "\n%s%s%s" (get_trans (-1006)) package_name
              (get_trans (-1008)) ;
          begin
          let answer = String.uppercase(read_line ()) in 
          handle_letter tp answer can_uninstall package_name m
                      finished !current ;
          Dc.clear_state () ; 
          end
        with e -> handle_error e
        done 
  end in 

  print_log () ; 

  let original_menu_style () = 
    let current = ref (-1) in 
    List.iter (fun m ->
      incr current ; 
      let def = ref (module_defaults.(!current)) in 
      let package_name = Dc.single_string_of_tlk_string game m.mod_name in 
      let can_uninstall = already_installed this_tp2_filename !current in 

      let preproc_fail msg warn already = 
        if (!def <> TP_Skip) && (!def <> TP_Uninstall) then begin
          (* WW: the "FORBID_FILE" bug -- don't uninstall things here! *)
        if (already) then 
            def := TP_Ask  
          else begin
            let warn = Dc.single_string_of_tlk_string game warn in 
            log_and_print "\n%s: [%s]\n\t%s\n" msg package_name warn ;
            def := TP_Skip 
          end 
        end 
      in 

      List.iter (fun f -> match f with
      | TPM_Deprecated(warn) -> 
        begin 
          if can_uninstall then begin
            let warn = Dc.single_string_of_tlk_string game warn in 
            log_and_print "\nNOTE: [%s] is deprecated. Uninstalling!\n\t%s\n" 
              package_name warn ;
            def := TP_Uninstall ; 
          end else begin
            def := TP_Skip ; 
          end 
        end 
      | TPM_RequireComponent(s,i,warn) -> 
        begin
          if already_installed s i && 
             not (temporarily_uninstalled s i) then
            () (* good! *) 
          else preproc_fail "SKIPPING" warn (already_installed s i)
        end 
      | TPM_ForbidComponent(s,i,warn) -> 
        begin
          if already_installed s i && 
             not (temporarily_uninstalled s i) then
            preproc_fail "SKIPPING" warn (already_installed s i)
          else 
            () (* good! *) 
        end 
      | TPM_RequirePredicate(p,warn) -> 
        begin
          if eval game p then 
            ()
          else preproc_fail "SKIPPING" warn can_uninstall
        end 
      | TPM_SubComponents(ts,_) -> () 
      | TPM_Designated(i) -> current := i 
      | TPM_NotInLog -> ()
      ) m.mod_flags ; 
      List.iter (fun a -> match a with
        | TP_Require_File(file,warn) ->
          begin
            if (file_exists file) then 
              ()
            else preproc_fail "SKIPPING" warn can_uninstall 
          end
        | TP_Forbid_File(file,warn) ->
          begin
            if (file_exists file) then 
              preproc_fail "SKIPPING" warn can_uninstall
            else 
              ()
          end
        | _ -> ()
      ) m.mod_parts ; 
      let finished = ref false in 
      let handle_error = handle_error_generic always_yes finished
        package_name in 
      Dc.clear_state () ; 
      match !def with
      | TP_Install ->
        begin try 
          handle_letter tp "Y" can_uninstall package_name m finished !current ;
          with e -> handle_error e  
        end 
      | TP_Uninstall ->
        begin try
          handle_letter tp "U" can_uninstall package_name m finished !current ;
          with e -> handle_error e
        end 
      | TP_Skip -> 
            (*
            log_and_print "\nSkipping Component [%s] %d\n" 
            (package_name) !current ; 
            *)
        begin
          match subcomp_group m with
          | Some(ts) -> Hashtbl.add asked_about_comp ts true  
          | None -> ()
        end
      | TP_Ask -> 
        begin
          match subcomp_group m with
          | Some(ts) -> ask_about_module_with_subcomp current m ts handle_error
          | None -> ask_about_module current m package_name handle_error
        end
    ) tp.module_list
  in 
  
  if !specify then begin
    let finished = ref false in
    while not !finished do
      log_and_print "\n%s" (get_trans (-1009)); 
      let line = read_line () in
      if line = "" then 
        finished := true
      else begin
        let reg = Str.regexp_string_case_fold line in 
        let current = ref (-1) in 
        List.iter (fun m ->
          incr current ;
          let rec process lst = match lst with
          | TPM_Designated(i) :: tl -> current := i ;
          | hd :: tl -> process tl
          | [] -> ()
          in process m.mod_flags ;
          let package_name = Dc.single_string_of_tlk_string game m.mod_name in 
          let handle_error = handle_error_generic (ref false) (ref true)
            package_name in 
          try 
            let _ = Str.search_forward reg package_name 0 in
            ask_about_module current m package_name handle_error
          with Not_found -> () 
        ) tp.module_list
      end 
    done 
  end else begin
    original_menu_style () ; 
  end ;

  interactive := false ; 

  (* now we must handle every temporarily-uninstalled mods *)
  let re_installed = ref [] in 
  let rec process lst = match lst with
    [] -> []
  | (_,_,_,_,Installed) as head :: tl -> head :: (process tl)
  | (_,_,_,_,Permanently_Uninstalled) as head :: tl -> head :: (process tl)
  | (a,b,c,sopt,Temporarily_Uninstalled) as head :: tl -> 
    begin 
    try 
      (* we must re-install it! *)
      begin
        (* log_and_print "\nRe-Installing [%s] component %d %s\n" 
          a c (str_of_str_opt sopt); *)
        log_and_print "\n%s%s%s %d %s\n" 
          (get_trans (-1010)) a (get_trans (-1011)) c 
          (str_of_str_opt sopt); 
        let tp_file = a in 
        let tp2 = handle_tp2_filename tp_file in 
        Load.allow_missing := !Load.allow_missing @ 
          List.fold_left (fun acc elt -> match elt with
            Allow_Missing(lst) -> lst @ acc
              | _ -> acc) [] tp2.flags ; 
        (* load their chosen language *)
        Dc.clear_state () ;
        Dc.push_trans ();
        init_default_strings () ; 
        (try
          let l = List.nth tp2.languages b in
          our_lang := Some(l) ;
          our_lang_index := b ; 
        (*  log_and_print "Re-Installing Using Language [%s]\n" l.lang_name ;*)
          log_and_print "%s [%s]\n" ((get_trans (-1012))) l.lang_name ; 
          Var.add_var "LANGUAGE" l.lang_dir_name ; 
          List.iter handle_tra_filename l.lang_tra_files
        with _ -> 
          our_lang := None ; 
          our_lang_index := 0 ;
          () ) ;
        let m = get_nth_module tp2 c in 
        let package_name = Dc.single_string_of_tlk_string game m.mod_name in 
        let fails_requirements = List.exists (fun f -> match f with 
          | TPM_RequireComponent(s,i,warn) -> 
            begin
              if already_installed s i then
                false
              else begin
                log_and_print "\n[%s] component %d %s fails component requirements, *not* Re-Installing.\n" a c (str_of_str_opt sopt); 
                true
              end 
            end
          | _ -> false) m.mod_flags 
        in 
        begin 
        if fails_requirements then begin 
          handle_letter tp2 "U" false package_name m (ref false) c ;
          re_installed := !re_installed @ [(a,b,c,sopt,Permanently_Uninstalled)] ;
        end else begin 
          handle_letter tp2 "R" false package_name m (ref false) c ;
          re_installed := !re_installed @ [(a,b,c,sopt,Installed)] ;
        end  
        end ; 
        Dc.clear_state () ; 
        Dc.pop_trans ();
      end ;
      (process tl)  
    with e ->
      log_and_print "ERROR Re-Installing [%s] component %d %s\nTry to re-install it manually.\n%s\n" a c (str_of_str_opt sopt) (Printexc.to_string e) ;
      (a,b,c,sopt,Permanently_Uninstalled) :: (process tl)
    end 
  in 
  let result = (process !the_log) @ !re_installed in 

  the_log := result ; 

  save_log game handle_tp2_filename handle_tra_filename ; 

  Load.allow_missing := old_allow_missing ; 
  game.Load.script_style <- old_script_style ; 
end
