(* generic utilities *)

open Arch
open Version 

type local_string_entry = {
  lse_male : string ;
  lse_male_sound : string ;
  lse_female : string ;
  lse_female_sound : string ;
} 

let log_channel = ref None 

let log_or_print fmt = 
  let k result = 
    match !log_channel with
      None -> print_string result ; flush stdout 
    | Some(o) -> output_string o result ; flush o 
  in 
  Printf.kprintf k fmt 

let log_only fmt = 
  let k result = 
    match !log_channel with
      None -> ()
    | Some(o) -> output_string o result ; flush o 
  in 
  Printf.kprintf k fmt 

let log_and_print fmt = 
  let k result = begin
    output_string stdout result ; flush stdout ;
    match !log_channel with
      None -> ()
    | Some(o) -> output_string o result ; flush o 
  end
  in 
  Printf.kprintf k fmt 

let inlined_files = Hashtbl.create 15 

let backup_ht = Hashtbl.create 511
let backup_dir = ref None 
let backup_list_chn = ref None

let set_backup_dir str i =
  let i = Printf.sprintf "%d" i in 
  let backup_dir_name = Printf.sprintf "%s/%s" str i in
  (try
    Unix.mkdir backup_dir_name 511 (* 511 = octal 0777 = a+rwx *)
  with _ -> ()) ;
  backup_dir := Some(backup_dir_name) ; 
  (match !backup_list_chn with
    Some(c) -> close_out c
  | None -> ()) ; 
  let backup_filename = (backup_dir_name ^ "/UNINSTALL." ^ i) in 
  Hashtbl.clear backup_ht ; 
  (try 
    backup_list_chn := Some(open_out_bin backup_filename)
  with e -> 
    log_and_print "WARNING: unable to open [%s]: %s
Will be unable to UNINSTALL later.\n" backup_filename (Printexc.to_string e))

let log_file = ref "" 

let init_log filename = 
  try
    let out = open_out filename in
    log_channel := Some(out) ;
    log_file := filename ;
    log_only "WeiDU v %s Log\n\n" version ;
    Array.iter (fun s -> log_only " %s" s) Sys.argv ;
    log_only "\n"  
  with e ->
    Printf.printf "WARNING: unable to open log file [%s]: %s"
      filename (Printexc.to_string e) ;
    () 

let int32_of_str_off str off =
  let d = Int32.of_int (Char.code str.[off+0]) in
  let c = Int32.of_int (Char.code str.[off+1]) in
  let b = Int32.of_int (Char.code str.[off+2]) in
  let a = Int32.of_int (Char.code str.[off+3]) in
  Int32.logor
  (Int32.logor (Int32.shift_left a 24)
              (Int32.shift_left b 16))
  (Int32.logor (Int32.shift_left c 8) (d))

let int_of_str_off str off =
  let d = Char.code str.[off+0] in
  let c = Char.code str.[off+1] in
  let b = Char.code str.[off+2] in
  let a = Char.code str.[off+3] in
  (a lsl 24) lor (b lsl 16) lor (c lsl 8) lor d

let int_of_str str = int_of_str_off str 0

let short_of_str_off str off =
  let d = Char.code str.[off+0] in
  let c = Char.code str.[off+1] in
  (c lsl 8) lor d

let byte_of_str_off str off =
  let d = Char.code str.[off] in
  d

let signed_byte_of d = 
  if d > 127 then 
    d - 256 
  else d

let short_of_str str = short_of_str_off str 0 

let str_of_int32 i =
  let twofivefive = Int32.of_int 255 in
  let d = Int32.to_int (Int32.logand i twofivefive) in
  let i = Int32.shift_right_logical i 8 in
  let c = Int32.to_int (Int32.logand i twofivefive) in
  let i = Int32.shift_right_logical i 8 in
  let b = Int32.to_int (Int32.logand i twofivefive) in
  let i = Int32.shift_right_logical i 8 in
  let a = Int32.to_int (Int32.logand i twofivefive) in
  let i = Int32.shift_right_logical i 8 in
  let result = String.make 4 (Char.chr a) in
  result.[0] <- (Char.chr d) ;
  result.[1] <- (Char.chr c) ;
  result.[2] <- (Char.chr b) ;
  result 

let str_of_int i =
  let d = i land 255 in
  let i = i lsr 8 in
  let c = i land 255 in
  let i = i lsr 8 in
  let b = i land 255 in
  let i = i lsr 8 in
  let a = i land 255 in
  let i = i lsr 8 in
  let result = String.make 4 (Char.chr a) in
  result.[0] <- (Char.chr d) ;
  result.[1] <- (Char.chr c) ;
  result.[2] <- (Char.chr b) ;
  result 

let str_of_short i =
  let d = i land 255 in
  let i = i lsr 8 in
  let c = i land 255 in
  let i = i lsr 8 in
  let result = String.make 2 (Char.chr d) in
  result.[1] <- (Char.chr c) ;
  result 

let str_to_exact_size str size =
  let dest = String.make size '\000' in
  let max = 
    if String.length str > size then
      size
    else
      String.length str
  in 
  String.blit str 0 dest 0 max ;
  dest

let write_int buff off value =
  String.blit (str_of_int (value)) 0 buff off 4  
let write_int32 buff off value =
  String.blit (str_of_int32 (value)) 0 buff off 4  
let write_short buff off value =
  String.blit (str_of_short (value)) 0 buff off 2  
let write_byte buff off value =
  if value < 0 then 
    buff.[off] <- (Char.chr (256+value))
  else 
    buff.[off] <- (Char.chr value) 
let write_resref buff off str =
  String.blit (str_to_exact_size str 8) 0 buff off 8  


let get_string_of_size str off size =
  let almost = String.sub str off size in
  try
    let null_index = String.index almost '\000' in
    String.sub almost 0 null_index
  with _ -> almost

let my_write size fd buff name = 
  let sofar = ref 0 in
  while !sofar < size do 
    let this_chunk = Unix.write fd buff !sofar (size - !sofar) in
    if this_chunk = 0 then begin
      failwith (Printf.sprintf "write %d of %d bytes from [%s]"
        !sofar size name) 
    end else 
      sofar := !sofar + this_chunk
  done 

let my_read size fd buff name = 
  let sofar = ref 0 in
  while !sofar < size do 
    let this_chunk = Unix.read fd buff !sofar (size - !sofar) in
    if this_chunk = 0 then begin
      failwith (Printf.sprintf "read %d of %d bytes from [%s]"
        !sofar size name) 
    end else 
      sofar := !sofar + this_chunk
  done 


let load_file name =
  if Hashtbl.mem inlined_files name then
    Hashtbl.find inlined_files name
  else 
  try begin
    Stats.time "loading files" (fun () -> 
    let stats = Unix.stat name in
    let size = stats.Unix.st_size in 
    if size = 0 then 
      log_or_print "WARNING: [%s] is a 0 byte file\n" name 
    else if size < 0 then begin
      log_and_print "ERROR: [%s] has reported size %d\n" name size ;
      failwith ("error loading " ^ name)
    end else if size > Sys.max_string_length then begin
      log_and_print "ERROR: [%s] has size %d: TOO BIG FOR WEIDU (max %d)\n" 
          name size Sys.max_string_length;
      failwith ("error loading " ^ name)
    end ; 
    let buff = String.make size '\000' in
    let fd = Unix.openfile name [Unix.O_RDONLY] 0 in
    my_read size fd buff name ; 
    Unix.close fd ; 
    log_only "[%s] loaded, %d bytes\n" name size ; 
    buff) () 
  end
  with e -> 
    log_and_print "ERROR: error loading [%s]\n" name ;
    raise e

let file_size name =
  try 
    let stats = Unix.stat name in
    stats.Unix.st_size 
  with _ ->  -1

let file_exists name = (file_size name >= 0) 

let is_directory name =
  try 
    let stats = Unix.stat name in
    stats.Unix.st_kind = Unix.S_DIR
  with _ -> false 

let split name =
  try 
    let base = Filename.chop_extension name in
    let ext = String.sub name ((String.length base)+1)
      ((String.length name) - ((String.length base)+1))
    in 
    base,ext
  with _ -> name,""

let list_of_files_in_directory d =
  let result = ref [] in 
  (try
    let dh = Unix.opendir d in
    begin
      try 
        while true do
          result := Unix.readdir dh :: !result 
        done 
      with _ -> () 
    end ;
    Unix.closedir dh 
  with _ -> ()) ; !result

let rec copy_directory source target =
  if not (is_directory target) then
  Unix.mkdir target 511 ;
  let dir_list = list_of_files_in_directory source in
    List.iter (fun d_or_f ->
      if d_or_f <> "." || d_or_f <> ".." then begin
		 Printf.printf "%s is the current dir\n" d_or_f ;
         if is_directory (source ^ "/" ^ d_or_f) then
            Unix.mkdir (target ^ "/" ^ d_or_f) 511 ;
            copy_directory (source ^ "/" ^ d_or_f) target
      end
    ) dir_list

let backup_if_extant filename = 
  if Hashtbl.mem backup_ht 
                 (String.uppercase (slash_to_backslash filename)) then 
    () 
  else begin 
    Hashtbl.add backup_ht
                (String.uppercase (slash_to_backslash filename)) true ;
    (
    match !backup_list_chn with
    | Some(chn) -> output_string chn (filename ^ "\n") ; flush chn 
    | None -> ()
    ); 
    match !backup_dir with
    | Some(dir) when file_exists filename -> 
      let backup_name = dir ^ "/" ^ (Filename.basename filename) in
      let buff = load_file filename in
      let backup_chn = open_out_bin backup_name in
      output_string backup_chn buff ;
      close_out backup_chn ;
      log_or_print "[%s] backed up to [%s]\n" filename backup_name 
    | _ -> () 
  end

let open_for_writing_internal backup filename binary = 
  (if (backup) then backup_if_extant filename) ;
  if file_exists filename then (* if it already exists *)
    begin (* handle read-only files! *)
      try 
        Unix.chmod filename 511 ; (* 511 = octal 0777 = a+rwx *)
      with e -> 
        log_or_print "WARNING: chmod %s : %s\n" filename 
          (Printexc.to_string e)
    end ;
  let out_chn = (if binary then open_out_bin else open_out) filename in
  out_chn 

let open_for_writing = open_for_writing_internal true 

let execute_at_exit = ref ([] : string list) 

(* for some stupid reason these cannot be in the parser or the lexer *)

type input_context = {
  mutable line : int ;
  mutable col  : int ;
  mutable delta : int ;
  mutable filename : string ;
  mutable lexbuf : Lexing.lexbuf ; 
  mutable warn_only : bool ; 
} 
let context_stack = ref [] 
let push_context filename lexbuf = 
  let new_context = { line = 1; col = 0; delta = 0; 
    filename = filename ; lexbuf = lexbuf ; warn_only = false } in
  context_stack := new_context :: !context_stack
let pop_context () = match !context_stack with
  [] -> log_and_print "ERROR: no current parsing context to pop!\n" ; () 
| hd::tl -> 
    context_stack := List.tl !context_stack
let the_context () = match !context_stack with
  hd :: tl -> hd
| [] -> log_and_print "ERROR: no current parsing context\n" ; 
        failwith "no current parsing context" 

let lex_init (file: string) 
         (inchannel: in_channel) : Lexing.lexbuf =
  let lexbuf = Lexing.from_channel inchannel in
  push_context file lexbuf ;
  lexbuf 

let lex_init_from_internal_string (file: string) 
         (buff: string) : Lexing.lexbuf =
  let lexbuf = Lexing.from_string buff in
  let ctx = the_context () in 
  let new_context = { line = ctx.line; col = ctx.col - ctx.delta; 
    delta = 0; filename = file ; lexbuf = lexbuf ; warn_only = true } in
  context_stack := new_context :: !context_stack ; 
  lexbuf 

let lex_init_from_string (file: string) 
         (buff: string) : Lexing.lexbuf =
  let lexbuf = Lexing.from_string buff in
  push_context file lexbuf ;
  lexbuf 

let newline () = 
  let c = the_context () in 
  c.line <- c.line + 1;
  c.col <- 1;
  c.delta <- 0

let tab () = 
  let c = the_context () in
  c.col <- c.col + 8 - (c.col mod 8)

let adj lb =
  let c = the_context () in 
  c.lexbuf <- lb ; 
  c.delta <- (Lexing.lexeme_end lb) - (Lexing.lexeme_start lb) ;
  c.col <- c.col + c.delta 

let str_adj lb =
  let c = the_context () in 
  c.lexbuf <- lb ; 
  let st= Lexing.lexeme lb in 
  for i = 0 to (String.length st) - 1 do
    if st.[i] = '\n' then newline () 
    else begin 
      c.col <- c.col + 1; c.delta <- c.delta + 1 ; 
    end
  done

let strip str = 
  let len = String.length str in
  String.sub str 1 (len - 2) 

let error_chn_ht = Hashtbl.create 11 
let error_chn_base = ref "iwg2/errors" 
let get_error_chn sort = 
  try
    Hashtbl.find error_chn_ht sort 
  with Not_found ->
    let oc = open_out (Printf.sprintf "%s/%s" !error_chn_base sort) in
    Hashtbl.add error_chn_ht sort oc ;
    oc

let error sort fmt = 
  let k result = 
    let oc = get_error_chn sort in
    output_string oc result ; 
    log_and_print "%s" result ; 
    flush oc ;
  in 
  Printf.kprintf k fmt 

let input_error_to_stdout = ref true 

let input_error sort_msg msg =
  let c = the_context () in 
  let near_text = Lexing.lexeme c.lexbuf in 
  (if !input_error_to_stdout then 
  log_and_print 
  else 
  error "DLG") 
  "\n[%s] %s %s at line %d column %d-%d\nNear Text: %s\n\t%s\n"
    c.filename 
    sort_msg
    (if c.warn_only then "WARNING" else "ERROR") 
    c.line (c.col - c.delta) (c.col-1) near_text msg ;
  raise Parsing.Parse_error

let lex_error msg = input_error "LEXER" msg 
let parse_error msg = input_error "PARSE" msg 

let my_int_of_string s = 
  try int_of_string s
  with e -> parse_error "Not An Integer"

(* big generic parsing function *) 
let parse_file filename sort_of_file parse_lex_fun =
  let do_the_work lexbuf =
    try 
      let result = Stats.time sort_of_file 
        (fun () -> parse_lex_fun lexbuf) () in
      pop_context () ;
      log_or_print "[%s] parsed\n" filename ; 
      result
    with e -> 
      (try input_error "" (Printexc.to_string e) with _ -> () ) ; 
      pop_context () ;
      raise e 
  in 
  if Hashtbl.mem inlined_files filename then begin
    let str = Hashtbl.find inlined_files filename in
    let lexbuf : Lexing.lexbuf = lex_init_from_string filename str in
    try 
      do_the_work lexbuf 
    with e -> 
      log_and_print "ERROR: parsing [%s]: %s\n" 
        filename (Printexc.to_string e) ; 
      raise e 
  end else begin 
    let inchan = open_in filename in 
    try 
      begin 
      let lexbuf : Lexing.lexbuf = lex_init filename inchan in
      let res = do_the_work lexbuf in
      close_in inchan ;
      res 
      end 
    with e -> 
      log_and_print "ERROR: parsing [%s]: %s\n" 
        filename (Printexc.to_string e) ; 
      close_in inchan ; raise e 
  end 
