/*
* Copyright 2006-2007 Multi Install Tool contributors. All Rights Reserved.
*
* You are permitted to redistribute this code under the terms of the GNU General
* Public License, version 2 or any later version. A copy of this license may be
* found in the file 'gpl.txt'.
*/
import java.util.List;
import java.io.File;
import java.io.FileFilter;
import javax.swing.JProgressBar;
import java.util.Map;
import java.io.UnsupportedEncodingException;
import java.io.FileWriter;
import java.io.BufferedWriter;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.ListIterator;

public final class CloneWorker extends org.jdesktop.swingworker.SwingWorker<Object, Object>{
	private CloningData data;
	private File source;
	private File sink;

	private boolean copyMusic;
	private boolean copyData;
	private boolean copyCharacters;
	private boolean copyPortraits;
	private boolean copySave;
	private MultiInstall.ProgressCallback callback;

	public CloneWorker(File source, File sink, CloningData data, 
	   boolean copyMusic, boolean copyData, boolean copyCharacters, 
	   boolean copyPortraits, boolean copySave,
	   MultiInstall.ProgressCallback callback){
		super();
		this.data = data;
		this.source = source;
		this.sink = sink;
		this.copyMusic = copyMusic;
		this.copyData = copyData;
		this.copyCharacters = copyCharacters;
		this.copyPortraits = copyPortraits;
		this.copySave = copySave;
		this.callback = callback;
	}

	private void doClone() throws Exception{
		callback.madeProgress("Preparing sink location", 0);
		sink.mkdirs();
		
		if(FileSystem.canLink()){
			FileSystem.link(source, sink);
		}
		List<String> requiredFiles = new LinkedList<String>();
		List<String> fakeFiles = new LinkedList<String>();
		Map<File, FileSystem.Patch> patches = 
		     new java.util.HashMap<File, FileSystem.Patch>();
		requiredFiles.addAll(Arrays.asList(data.files));
		requiredFiles.addAll(Arrays.asList(CloningData.commonFiles));
		requiredFiles.add(data.exeName);
		if(data.iniName != null){
			requiredFiles.add(data.iniName);
		}

		if(fileexists("dialogf.tlk")){
			requiredFiles.add("dialogf.tlk");
		}
		if(copySave && fileexists("save")){
			requiredFiles.add("save");
		}
		if(!FileSystem.canLink()){
			if(copyPortraits && fileexists("portraits")){
				requiredFiles.add("portraits");
			}
			if(copyCharacters && fileexists("characters")){
				requiredFiles.add("characters");
			}
			if(copyData && fileexists("data")){
				requiredFiles.add("data");
				if(fileexists("movies")){
					requiredFiles.add("movies");
				}
				copyMusic = true;
			}
			else{
				for(String s : new File(source, "data").list()){
					fakeFiles.add("data" + File.separator + s);
				}
			}
			if(copyMusic && fileexists("music")){
				requiredFiles.add("music");
			}


			patches.put(
				source.listFiles(
					new CaseInsensitiveFileFilter(data.exeName)
				)[0],
				new FileSystem.BinaryPatch(){
					public int[] getPatchableOffsets(){
						if(copyData){
							return new int[0];
						}
						List<Integer> offsets = 
						  new java.util.ArrayList<Integer>(3);
						offsets.add(data.maleTlkOffset);
						offsets.add(data.femaleTlkOffset);
						if(copyMusic){
							offsets.add(data.musicOffset);
						}
						int[] ret = new int[offsets.size()];
						for(int i = 0; i < ret.length; ++i){
							ret[i] = offsets.get(i);
						}
						return ret;
					}
		
					public byte[] getOldValueAt(int offset){
						try{
							if(offset == data.maleTlkOffset){
								return "hd0:\\dialog.tlk".getBytes(
									"US-ASCII"
								);
							}
							else if(offset == data.femaleTlkOffset){
								return "hd0:\\dialogf.tlk".getBytes(
									"US-ASCII"
								);
							}
							else if(offset == data.musicOffset){
								return "hd0:\\music".getBytes(
									"US-ASCII"
								);
							}
						}
						// Can't happen - US-ASCII is always supported
						catch(UnsupportedEncodingException uee){}
						return null;
					}
		
					public byte[] getNewValueAt(int offset){
						try{
							if(offset == data.maleTlkOffset){
								return ".\\dialog.tlk\0\0\0".getBytes(
									"US-ASCII"
								);
							}
							else if(offset == data.femaleTlkOffset){
								return ".\\dialogf.tlk\0\0\0".getBytes(
									"US-ASCII"
								);
							}
							else if(offset == data.musicOffset){
								return ".\\music\0\0\0".getBytes(
									"US-ASCII"
								);
							}
						}
						// Can't happen - US-ASCII is always supported
						catch(UnsupportedEncodingException uee){}
						return null;
					}
		
					public Map<String, String> getTextReplacements(){
						return new java.util.HashMap<String, String>();
					}
				}
			);
	
			patches.put(
				source.listFiles(
					new CaseInsensitiveFileFilter(
						data.iniName
					)
				)[0],
				
				new FileSystem.TextPatch(){
					public Map<String, String> getReplacements(){
						Map<String, String> ret = 
						  new java.util.HashMap<String, String>();
						try{	
							if(copyData) ret.put(
								source.getCanonicalPath(),
								sink.getCanonicalPath()
							);
						}
						catch(java.io.IOException ioe){}
						return ret;
					}
				}
			);
		}
		else{
			/* 
			 * If we can link, we don't need to copy any directories 
			 * ever. There may be problems with things copying to the
			 * link target, instead of overwriting the link, but 
			 * apps are meant to check things like that anyway.
			 */
			for(ListIterator it = requiredFiles.listIterator(); it.hasNext(); ){
				String filename = it.next().toString();
				if(new File(source, filename).isDirectory()){
					it.remove();
				}
			}
		}

		final int numFiles = requiredFiles.size() + fakeFiles.size();
		for(int i = 0; i < requiredFiles.size(); ++i){
			if(isCancelled()){
				return;
			}

			callback.madeProgress(
				"Copying " + requiredFiles.get(i),
				i * 100 / numFiles
			);

			File sourceFile;
			try{
				sourceFile = source.listFiles(
					new CaseInsensitiveFileFilter(
						requiredFiles.get(i)
					)
				)[0];
			}
			catch(ArrayIndexOutOfBoundsException e){
				throw new java.io.FileNotFoundException(
					requiredFiles.get(i).toString()
				);
			}

			FileSystem.Patch patch = FileSystem.COPY_VERBATIM;
			if(patches.containsKey(sourceFile)){
				patch = patches.get(sourceFile);
			}
			FileSystem.copy(
				sourceFile,
				sink,
				patch
			);
		}
		for(int i = 0; i < fakeFiles.size(); ++i){
			if(isCancelled()){
				return;
			}
			callback.madeProgress(
				"Writing nonsense instead of copying " + fakeFiles.get(i),
				(i + requiredFiles.size()) * 100 / numFiles
			);
			File f = new File(sink, fakeFiles.get(i));
			f.getParentFile().mkdirs();
			BufferedWriter bw = new BufferedWriter(new FileWriter(f));
			for(String msg : new String[]{
				"TO WHOM IT MAY CONCERN",
				"Hello.",
				"Regards,",
				"The MultiInstall Tool"
			}){
				bw.write(msg);
				bw.newLine();
			}
			bw.flush(); bw.close();
		}
	}


	public Object doInBackground() throws Exception{
		try{
			doClone();
			callback.finished();
		}
		catch(Exception e){
			callback.reportException(e);
		}
		return null;
	}

	private boolean fileexists(String filename){
		return source.listFiles(new CaseInsensitiveFileFilter(filename)).length > 0;
	}
}
