/*
 * Copyright 2006-2007 Multi Install Tool contributors. All Rights Reserved.
 *
 * You are permitted to redistribute this code under the terms of the GNU General
 * Public License, version 2 or any later version. A copy of this license may be
 * found in the file 'gpl.txt'.
 */

import java.io.*;
import java.util.*;

/**
 * Provides methods for dealing with files in the context of a file system.
 * @author Lennon Victor Cook
 * @date 2006 - 07 - 10 
 */
public class FileSystem{
	private static final int BUFFSIZE = 30 * 1024 * 1024;

	/**
	 * A PatchingModel which will cause copy() to copy the source
	 * verbatim to sink.
	 */
	public static final Patch COPY_VERBATIM = new Patch(){};

	/**
	 * Copies source into sink. May cause undesirable results when sink
	 * is a symlink.
	 * @param source the file to copy. If source is a directory, it is
	 * copied recursively using the given Patch for all files.
	 * @param sink the location to copy it to. If sink is a directory,
	 * source is copied into that directory; otherwise, source is copied
	 * byte-for-byte to sink, overwriting sink if it exists, and creating
	 * it if it doesn't.
	 * @throws IOException if an I/O error occured.
	 * @throws FileNotFoundException if source could not be found, is 
	 * a directory, or could not be opened for any other reason.
	 * @throws PatchFailedException if the patching failed for any other
	 * reason.
	 */
	public static void copy(File source, File sink, Patch patch)
					throws 	IOException,
						FileNotFoundException,
					 	PatchFailedException{
		// Copy into directory instead of over it.
		if(sink.isDirectory()){
			copy(source, new File(sink, source.getName()), patch);
			return;
		}
		// Copy source directories recursively
		if(source.isDirectory()){
			if(sink.exists()&& !sink.isDirectory()){
				throw new IOException();	
			}
			sink.mkdirs();
			for(File f : source.listFiles()){
				copy(f, sink, patch);
			}
			return;
		}

		/* 
		 * Create sinks parent dirs if they don't exist, lest
		 * the streams complain.
		 */ 
		if(!sink.exists() && sink.getParentFile() != null){
			sink.getParentFile().mkdirs();
		}

		if(patch instanceof BinaryPatch){
			copy_binary(source, sink, (BinaryPatch) patch);
		}
		else if(patch instanceof TextPatch){
			copy_textually(source, sink, (TextPatch) patch);
		}
		else{
			patch = new BinaryPatch(){
				public int[] getPatchableOffsets(){
					return new int[0];
				}
		
				public byte[] getOldValueAt(int offset){
					return new byte[0];
				}
		
				public byte[] getNewValueAt(int offset){
					return new byte[0];
				}
		
				public Map<String,String> getTextReplacements(){
					return new HashMap<String,String>();		
				}
			};
			copy_binary(source, sink, (BinaryPatch) patch);
		}
	}

	private static void copy_binary(File source, File sink, BinaryPatch patch)
	                           throws	IOException,
	                                 	FileNotFoundException,
		                         	PatchFailedException{
		InputStream in = new FileInputStream(source);
		OutputStream out = new FileOutputStream(sink);
		
		// Generate [offset0, offset1 ... offsetn, file length]
		// Curse that Java arrays are fixed-length
		int[] initialOffsets = patch.getPatchableOffsets();
		int[] offsets = new int[initialOffsets.length + 1];
		for(int i = 0; i < initialOffsets.length; ++i){
			offsets[i] = initialOffsets[i];
		}
		offsets[initialOffsets.length] = (int) source.length();
		
		int currentOffset = 0;

		Arrays.sort(offsets);
		for(int offset : offsets){
			// Copy everything before the patch verbatim, in chunks
			int difference;
			while((difference = offset - currentOffset) > 0){
				int size = (difference > BUFFSIZE ? BUFFSIZE : difference); 
				byte[] buff = new byte[size];
				in.read(buff);
				out.write(buff);
				currentOffset += size;
			}
			// If we're at the end, don't try to do a patch
			if(offset == (int) source.length()){
				break;
			}

			// Then replace the old value with the new.
			byte[] oldVal = patch.getOldValueAt(offset);
			byte[] newVal = patch.getNewValueAt(offset);
			byte[] buff   = new byte[oldVal.length];
			
			in.read(buff);
			// Don't fail for already-patched files
			if(Arrays.equals(buff, oldVal) || Arrays.equals(buff, newVal)){
				out.write(newVal);
				currentOffset += buff.length;
			}
			else{
				throw new PatchFailedException(
					source.getName(),
					oldVal,
					buff
				);
			}
		}
		in.close();
		out.close();
	}

	private static void copy_textually(File source, File sink, TextPatch patch)
	                           throws	IOException,
	                                 	FileNotFoundException{
		/* 
		 * Create tempfile so we're writing to different 
		 * files, to avoid zeroing out anything.
		 */
		BufferedReader in = new BufferedReader(new FileReader(source));
		BufferedWriter out = new BufferedWriter(new FileWriter(sink));
	
		Map<String,String> textReplacements =  patch.getReplacements();
		String buff;
		while((buff = in.readLine()) != null){
			for(String s : textReplacements.keySet()){
				buff.replace(s, textReplacements.get(s));
			}
			out.write(buff);
			out.newLine();
		}
		in.close();
		out.close();
	}

	private static boolean checkedLinkingBefore = false;
	private static boolean canLink;
	public static boolean canLink(){
		if(!checkedLinkingBefore){
			checkedLinkingBefore = true;
			try{
				Runtime.getRuntime().exec(new String[]{"ln"});
				canLink = true;
			}
			catch(Exception e){
				canLink = false;
			}
		}
		return canLink;
	}

	/**
	 * Lists every non-directory file found in the given directory. Any sub-
	 * directories found are replaced with the output of this function on them,
	 * prepended with the name of the original directory and File.separator 
	 */
	public static String[] listAllFiles(File dir){
		LinkedList<String> ret = new LinkedList<String>();
		for(File f : dir.listFiles()){
			if(f.isDirectory()){
				for(String s : listAllFiles(f)){
					ret.add(dir.getName() + File.separator + s);
				}
			}
			else{
				ret.add(dir.getName() + File.separator + f.getName());
			}
		}
		return ret.toArray(new String[0]);
	}

	/**
	 * Convenience method - precisely equivalent to
	 * <code>link(source, sink, true);</code>.
	 */
	public static void link(File source, File sink) throws IOException{
		link(source, sink, true);	
	}

	/**
	 * Symlinks the source to the sink. Outsources work to ln(1).
	 * @param source the file to link from.
	 * @param sink the location to link it to.
	 * @param recurseDirs if true, and source is a directory, then it
	 * is linked recursively, per lndir(1).
	 * @throws IOException if an I/O error occured. 
	 */
	public static void link(File source, File sink, boolean recurseDirs)
					throws IOException{
		if(source.isDirectory() && recurseDirs){
			sink = new File(sink, source.getName());
			sink.mkdir();
			for(File f : source.listFiles()){
				link(f, sink, true);
			}
		}
		else{
			Runtime.getRuntime().exec(new String[]{
				"ln",
				"-s",
				source.getPath(),
				sink.getPath()
			});
		}
	}

	/** Parent interface for copy() convenience */
	public static interface Patch{}

	/**
	 * Provides copy() with the information it needs to patch a binary file
	 * as it copies. All methods are guaranteed to have no side-effects.
	 */
	public static interface BinaryPatch extends Patch{
		/**
		 * @return an array containing all the offsets which should
		 * be patched.
		 */
		int[] getPatchableOffsets();

		/**
		 * @return the value that should be in the source file at
		 * offset.
		 */
		byte[] getOldValueAt(int offset);

		/**
		 * @return the value that should be put in the sink file at
		 * offset.
		 */
		byte[] getNewValueAt(int offset);


	}

	public static interface TextPatch extends Patch{
		/**
		 * Gets text-wise replacements. All occurances of a key String
		 * in the file should be replaced with its corresponding value
		 * String, except when strings are split across lines.
		 */
		Map<String,String> getReplacements();
	}

	public static final class PatchFailedException extends Exception{
		public final String expectedValue;
		public final String gotValue;
		public final String filename;

		public PatchFailedException(	String filename, 
						byte[] expectedValue, 
						byte[] gotValue
		){
			super();
			this.filename = filename;
			this.expectedValue = new String(expectedValue);
			this.gotValue = new String(gotValue);
		}

		public String toString(){
			return "PatchFailedException in " + filename 
				+ " expected " + expectedValue + " got "
				 	+ gotValue;
		}
	}
}

