/*
 * Copyright 2006-2007 Multi Install Tool contributors. All Rights Reserved.
 *
 * You are permitted to redistribute this code under the terms of the GNU General
 * Public License, version 2 or any later version. A copy of this license may be
 * found in the file 'gpl.txt'.
 */

import javax.swing.*;
import java.awt.Component;
import java.awt.event.*;
import javax.swing.event.*;

import java.io.IOException;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.FileNotFoundException;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;

import jargs.gnu.*;

/**
 * @author Lennon Victor Cook
 * @date 2006-07-16 
 */
public final class MultiInstall{
	static boolean verbose;

	public static void main(String[] args){
		CmdLineParser parser = new CmdLineParser();
		CmdLineParser.Option batch       = parser.addBooleanOption("batch");
		CmdLineParser.Option sourceO     = parser.addStringOption("source");
		CmdLineParser.Option sinkO       = parser.addStringOption("sink");
		CmdLineParser.Option musicO      = parser.addBooleanOption("copy-music");
		CmdLineParser.Option charactersO = parser.addBooleanOption("copy-characters");
		CmdLineParser.Option dataO       = parser.addBooleanOption("copy-data");
		CmdLineParser.Option saveO       = parser.addBooleanOption("copy-save");
		CmdLineParser.Option portraitsO  = parser.addBooleanOption("copy-portraits");
		CmdLineParser.Option verboseO    = parser.addBooleanOption('v', "verbose");
		
		try{
			parser.parse(args);
		}
		catch(CmdLineParser.OptionException e){
			System.err.println(e.getMessage());
			printUsage();
			System.exit(2);
		}
		String source = (String) parser.getOptionValue(sourceO, "");
		String sink   = (String) parser.getOptionValue(sinkO, "");
		Boolean music = (Boolean) parser.getOptionValue(musicO, false);
		Boolean characters = (Boolean) parser.getOptionValue(charactersO, false);
		Boolean data  = (Boolean) parser.getOptionValue(dataO, false);
		Boolean save  = (Boolean) parser.getOptionValue(saveO, false);
		Boolean portraits = (Boolean) parser.getOptionValue(portraitsO, false);
		verbose        =  (Boolean) parser.getOptionValue(verboseO, false); 

		try{
			if((Boolean) parser.getOptionValue(batch, false)){
				CloningData cd = CloningData.instanceFor(new File(source));
				CloneWorker cw = new CloneWorker(new File(source), 
				   new File(sink), cd, music, data, characters,
				   portraits, save, batchCallback);
				cw.execute();
				// Block until the clone finishes
				cw.get();	
			}
			else{
				new MultiInstall(source, sink, 
				  music, characters, data, save, portraits);
			}
		}
		catch(Exception e){
			e.printStackTrace();
			System.exit(1);
		}
	}

	private static void printUsage(){
		// Yeah, it's a copout, but the readme *does* actually have this
		System.err.println("Please see the readme for usage information.");
	}

	public MultiInstall(String sourcepath, String sinkpath, boolean copyMusic,
	   boolean copyCharacters, boolean copyData, boolean copySave, 
	   boolean copyPortraits) throws Exception{
		try{
			UIManager.setLookAndFeel(
				UIManager.getSystemLookAndFeelClassName()
			);
		}
		catch(Exception e){}
		
		new MultiGUI(this);
		source.addCaretListener(updateSource);
		sink.addCaretListener(updateSink);
		source.setText(sourcepath);
		sink.setText(sinkpath);
		music.setSelected(copyMusic);
		characters.setSelected(copyCharacters);
		data.setSelected(copyData);
		save.setSelected(copySave);
		portraits.setSelected(copyPortraits);
		// Get everything working
		updateSource.caretUpdate(null);
		window.pack();
		window.setVisible(true);
	}

	private CloningData clonedata = null;
	private CloneWorker worker = null;

	private FileFilter musicFilter = new CaseInsensitiveFileFilter("music");
	private FileFilter dataFilter = new CaseInsensitiveFileFilter("data");
	private FileFilter portraitsFilter = new CaseInsensitiveFileFilter("portraits");
	private FileFilter charactersFilter = new CaseInsensitiveFileFilter("characters");
	private FileFilter saveFilter = new CaseInsensitiveFileFilter("save");

	/* Widgets to get initialised by SwiXML */
	public JTextField source;
	public JTextField sink;
	public JProgressBar progress;
	public JFrame window;
	public JCheckBox music;
	public JCheckBox data;
	public JCheckBox portraits;
	public JCheckBox characters;
	public JCheckBox save;

	/* Actions, assigned to appropriate widgets by SwiXML */
	public Action browseForSource = new AbstractAction(){
		private JFileChooser jfc = new JFileChooser();

		public void actionPerformed(ActionEvent ae){
			jfc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			if(jfc.showOpenDialog(window) == jfc.APPROVE_OPTION){
				source.setText(jfc.getSelectedFile().getPath());
			}
		}
	};

	public Action browseForSink= new AbstractAction(){
		private JFileChooser jfc = new JFileChooser();

		public void actionPerformed(ActionEvent ae){
			jfc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			if(jfc.showOpenDialog(window) == jfc.APPROVE_OPTION){
				sink.setText(jfc.getSelectedFile().getPath());
			}
		}
	};


	public Action doClone = new AbstractAction(){
		public void actionPerformed(ActionEvent ae){
			boolean copyPortraits = portraits.isEnabled() 
				&& portraits.isSelected();
			boolean copyMusic = music.isEnabled() && music.isSelected();
			boolean copyData = data.isEnabled() && data.isSelected();
			boolean copySave = save.isEnabled() && save.isSelected();
			boolean copyCharacters = characters.isEnabled()
				&& characters.isSelected();

			// Temporarily disable all components
			source.setEnabled(false);
			browseForSource.setEnabled(false);
			sink.setEnabled(false);
			browseForSink.setEnabled(false);
			doClone.setEnabled(false);
			music.setEnabled(false);
			data.setEnabled(false);
			characters.setEnabled(false);
			portraits.setEnabled(false);
			save.setEnabled(false);
			
			// Enable cancelling now it makes sense
			cancel.setEnabled(true);

			worker = new CloneWorker(
				new File(source.getText()),
				new File(sink.getText()),
				clonedata,
				copyMusic,
				copyData,
				copyCharacters,
				copyPortraits,
				copySave,
				interactiveCallback
			);

			worker.addPropertyChangeListener(progressUpdater);
			worker.execute();
		}
	};

	public Action cancel = new AbstractAction(){
		{
			setEnabled(false);
		}

		public void actionPerformed(ActionEvent ae){
			while(!worker.cancel(true));
		}
	};

	/* Listeners added manually */
	
	public CaretListener updateSource = new CaretListener(){
		public void caretUpdate(CaretEvent ce){
			File sourceFile = new File(source.getText());

			try{
				clonedata = CloningData.instanceFor(sourceFile);
			}
			catch(Exception e){
				clonedata = null;
			}

			music.setEnabled(
				clonedata != null
				&& clonedata.musicOffset != -1
				&& sourceFile.listFiles(musicFilter).length != 0
				&& !FileSystem.canLink()
			);
			data.setEnabled(
				clonedata != null
				&& sourceFile.listFiles(dataFilter).length != 0
				&& !FileSystem.canLink()
			);
			portraits.setEnabled(
				clonedata != null
				&& sourceFile.listFiles(portraitsFilter).length != 0
				&& !FileSystem.canLink()
			);
			characters.setEnabled(
				clonedata != null
				&& sourceFile.listFiles(musicFilter).length != 0
				&& !FileSystem.canLink()
			);
			save.setEnabled(
				clonedata != null
				&& sourceFile.listFiles(saveFilter).length != 0
			);
			// Enable 'Go' if the sink is right too.
			updateSink.caretUpdate(ce);
		}
	};

	public CaretListener updateSink = new CaretListener(){
		public void caretUpdate(CaretEvent ce){
			File sinkFile = new File(sink.getText());
			doClone.setEnabled(
				clonedata != null && (
					   sinkFile.isDirectory()
					|| !sinkFile.exists()
				)
			);
		}
	};

	final PropertyChangeListener progressUpdater = new PropertyChangeListener(){
		public void propertyChange(PropertyChangeEvent evt){
			try{
				int val = (Integer) evt.getNewValue();
				progress.setValue(val);
				progress.revalidate();
			}
			catch(ClassCastException e){}
		}
	};

	/** Reset components when worker is done or cancelled */
	void reenableGUI(){
		if( worker == null || !worker.isDone()){
			return;
		}
		worker.removePropertyChangeListener(progressUpdater);
		worker = null;
		cancel.setEnabled(false);
		source.setEnabled(true);
		browseForSource.setEnabled(true);
		sink.setEnabled(true);
		browseForSink.setEnabled(true);
		updateSource.caretUpdate(null);
	}

	static void logException(Exception e){
		try{
			
			e.printStackTrace(
				new PrintWriter(new FileOutputStream(
					"MultiInstallTool.err",
					false
				))
			);
		}
		catch(FileNotFoundException f){}
		e.printStackTrace();		
	}

	private static ProgressCallback batchCallback = new ProgressCallback(){
		public void madeProgress(String newString, int progressPercent){
			if(verbose){
				System.out.println(progressPercent + "%" + " :" + newString);
			}
		}

		public void reportException(Exception e){
			logException(e);
			System.exit(3);
		}
		public void finished(){}
		
	};

	// Use callbacks for logging?
	private ProgressCallback interactiveCallback = new ProgressCallback(){
		public void madeProgress(String newString, int progressPercent){
			progress.setString(newString);
			progress.setValue(progressPercent);
			batchCallback.madeProgress(newString, progressPercent);
		}
	
		public void finished(){
			madeProgress("Done", 100);
			reenableGUI();
		}

		public void reportException(Exception e){
			JOptionPane.showMessageDialog(
				window,
				e.toString()
			);
			worker.cancel(true);
			logException(e);
			progress.setString("Cancelled");
		}

	};

	public interface ProgressCallback{
		public void reportException(Exception e);
		public void madeProgress(String newString, int progressPercent);
		public void finished();
	}
}
