/* Set TABS = 4 */
/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OggVorbis SOFTWARE CODEC SOURCE CODE.   *
 * USE, DISTRIBUTION AND REPRODUCTION OF THIS LIBRARY SOURCE IS     *
 * GOVERNED BY A BSD-STYLE SOURCE LICENSE INCLUDED WITH THIS SOURCE *
 * IN 'COPYING'. PLEASE READ THESE TERMS BEFORE DISTRIBUTING.       *
 *                                                                  *
 * THE OggVorbis SOURCE CODE IS (C) COPYRIGHT 1994-2001             *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *
 *                                                                  *
 ********************************************************************

 function: Decoder/player using vorbisfile, VorbisGain.
 last mod: $Id: OggDec decoder by John Edwards last updated 2002-07-17

 ********************************************************************/

/* Takes a vorbis bitstream from input ogg files and writes stereo PCM to
 * output wave files using vorbisfile, VorbisGain. 
 * Multiple input files are permitted, with wildcard use. Use is made of 
 * VorbisGain tags during decoding, if required and if the correct tags 
 * are present (old or new). Absence of the tags, even if their use is 
 * requested, does not cause an error; decoding proceeds with no gain
 * adjustment. Output maybe 8 bit, 16 bit, 24 bit or 32 bit integer, or 
 * 32 bit float. VorbisGain scaling is applied to the decoded floats 
 * before formatting the output.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <vorbis/float_math.h>
#include <string.h>
#include <ctype.h>
#include <windows.h>
#include <time.h>

#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>
#include <getopt.h>

#include "audio.h"

#include "wave_out.h"           // included WIN_AUDIO

#include "vcedit.h"             // included for vorbis tag reading routines

#include "dither.h"             // included for dither functions

#define READ_SIZE 4096          // length of samples to be read

#define ROUND64(x)   ( doubletmp = (x) + Dither.Add + (__int64)0x001FFFFD80000000L, *(__int64*)(&doubletmp) - (__int64)0x433FFFFD80000000L )

typedef struct file_list
{
	char *file_name;
	struct file_list *next_file;
} FILE_LIST;

/* Declare Functions */
static void usage(void);
void file_error(const char *message, const char *filename);
void free_file_list(FILE_LIST *list, int free_names);
char **glob(char *files[], int *count, int shuffle);
void free_glob(char *file_names[]);
static int tag_compare(const char *s1, const char *s2, int n);
double get_scale(const char *filename, int album, int title);

dither_t            Dither;
double              doubletmp;

struct option long_options[] = {
	{"help",0,0,'h'},
	{"album",0,0,'a'},
	{"radio",0,0,'r'},
	{"dither",1,0,'d'},
	{"play",0,0,'p'},
	{"shuffle",0,0,'s'},
	{"stdout",0,0,'o'},
	{"class",1,0,'c'},
	{"bits",1,0,'b'},
	{NULL,0,0,0}
};
	
/* Format a file error message, using the supplied file name and the message 
 * for the error in GetLastError(), in that order, as printf arguments for 
 * the message body.
 */
void file_error(const char *message, const char *filename)
{
	LPVOID error;
    
	FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | 
		FORMAT_MESSAGE_IGNORE_INSERTS, NULL, GetLastError(), 
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR) &error, 0, 
		NULL);
    
	fprintf(stderr, message, filename, error);
	LocalFree(error);
}

/* Free a FILE_LIST list. If free_names is true, free the name 
 * in each list entry as well.
 */
void free_file_list(FILE_LIST *list, int free_names)
{
	FILE_LIST *next;
    
	while (list)
	{
		next = list->next_file;
        
		if (free_names)
		{
			free(list->file_name);
		}
        
		free(list);
		list = next;
	}
}


/* Expand any wildcards in the files array (with count entries). The 
 * returned array is null terminated, and count is set to the number 
 * of entries, excluding the null. If NULL is returned, an error has 
 * occured and a suitable error message has been printed.
 *
 * Call free_glob to free the memory allocated by this function.
 */
char **glob(char *files[], int *count, int shuffle)
{
	FILE_LIST         *first = NULL,
	                  *current,
	                  *previous = NULL;
	WIN32_FIND_DATA   find_data;
	HANDLE            *find_handle;
	char              **file_names,
	                  *tmp;
	int               file_count = 0,
	                  path_length = 0,
	                  i,
	                  j;
    
	for (i = 0; i < *count; ++i)
	{
		for (j = 0; j < strlen(files[i]); j++)
		{
			if (files[i][j] == '\\')
			{
				path_length = j + 1;
			}
		}
        
		find_handle = FindFirstFile(files[i], &find_data);
        
		if (INVALID_HANDLE_VALUE == find_handle)
		{
			file_error(" Could not find '%s': %s", files[i]);
			files[i] = NULL;
			continue;
		}
        
		do
		{
			current = (struct file_list *) calloc(1, sizeof(struct file_list));
			current->file_name = calloc(path_length + strlen(find_data.cFileName) + 1, 1);
            
			if (path_length)
			{
				strncpy(current->file_name, files[i], path_length);
				strcat(current->file_name, find_data.cFileName);
			}
			else
			{
				strcpy(current->file_name, find_data.cFileName);
			}
            
			if (previous != NULL)
			{
				previous->next_file = current;
			}
            
			if (first == NULL)
			{
				first = current;
			}
            
			previous = current;
			file_count++;
		} while (FindNextFile(find_handle, &find_data));
        
		if (GetLastError() != ERROR_NO_MORE_FILES)
		{
			file_error(" Couldn't get more files for '%s': %s\n", files[i]);
			free_file_list(first, 1);
			files[i] = NULL;
		}
        
		/* Can return error, but do we need to bother? */
		FindClose(find_handle);
	}
    
	file_names = (char **) calloc(file_count + 1, sizeof(char *));
    
	for (i = 0, current = first; current; current = current->next_file, i++)
	{
		file_names[i] = current->file_name;
	}
    
	/* set up playlist */

	if (shuffle)
	{
		srand(time(0));
		/* initial shuffle */
		for (i = 0; i < file_count; ++i)
		{
			j = rand() % file_count;

			tmp = file_names[i];
			file_names[i] = file_names[j];
			file_names[j] = tmp;
		}
	}
	free_file_list(first, 0);
	*count = file_count;
	return file_names;
}


/* Free memory allocated by glob. */
void free_glob(char *file_names[])
{
	char **current;
	int i = 0;

	for (current = file_names; *current; current++)
	{
		free(*current);
	}
    
	free(file_names);
}

/* Dither output */
__int64 dither_output(int dithering, int shapingtype, long i, double Sum, int k)
{
	double Sum2;
	__int64 val;
	if(dithering)
	{
		if(!shapingtype)
		{
			double  tmp = Random_Equi ( Dither.Dither );
			Sum2 = tmp - Dither.LastRandomNumber [k];
			Dither.LastRandomNumber [k] = tmp;
			Sum2 = Sum += Sum2;
			val = ROUND64 (Sum2)  &  Dither.Mask;
		}
		else
		{
			Sum2  = Random_Triangular ( Dither.Dither ) - scalar16 ( Dither.DitherHistory[k], Dither.FilterCoeff + i );
			Sum  += Dither.DitherHistory [k] [(-1-i)&15] = Sum2;
			Sum2  = Sum + scalar16 ( Dither.ErrorHistory [k], Dither.FilterCoeff + i );
			val = ROUND64 (Sum2)  &  Dither.Mask;
			Dither.ErrorHistory [k] [(-1-i)&15] = (float)(Sum - val);
		}
		return (val);
	}
	else
		return (ROUND64 (Sum));
}

/* This is more or less the same as strncasecmp - but that doesn't exist
 * everywhere, and this is a fairly trivial function, so we include it 
 */
static int tag_compare(const char *s1, const char *s2, int n)
{
	int c;

	for (c = 0; c < n; c++)
	{
		if (toupper(s1[c]) != toupper(s2[c]))
		{
			return toupper(s1[c]) - toupper(s2[c]);
		}
	}

	return 0;
}


/* Reads ReplayGain tags from the input file. If audiophile is true, 
 * checks for audiophile tag, otherwise it reads the radio tag. If
 * the tags are found, it returns the scale value calculated from the
 * tag, otherwise, it returns a scale value of 1.0.
 */
double get_scale(const char *filename, int album, int title)
{
	vcedit_state    *state = NULL;
	vorbis_comment  *vc;
	FILE            *file = NULL;
	int             i,
	                j,
	                k;
	char            tag_buf[15];
	double          gain_A = 0,
	                gain_R = 0,
	                peak_A = 0,
	                peak_R = 0,
	                scale_factor = 1,
	                factor_clip;

	state = vcedit_new_state();

	if (state == NULL)
	{
		fprintf(stderr, "Out of memory\n");
		goto exit;
	}

	file = fopen(filename, "rb");

	if (file == NULL)
	{
		fprintf(stderr, "Couldn't open file '%s'\n", filename);
		goto exit;
	}

	if (vcedit_open(state, file) < 0)
	{
		fprintf(stderr, "Couldn't open file '%s' as vorbis: %s\n",
				filename, vcedit_error(state));
		goto exit;
	}

	vc = vcedit_comments(state);

	for (i = 0; i < vc->comments; i++)
	{ 
		/* Check for ReplayGain tags and get the peak and gain values. */
		if (!tag_compare(vc->user_comments[i], "RG_PEAK=", 8))
		{
			for(j = 8, k = 0; j <= strlen(vc->user_comments[i]); j++, k++)
				tag_buf[k] = vc->user_comments[i][j];
			peak_R = atof(tag_buf);
			continue;
		}
		if (!tag_compare(vc->user_comments[i], "REPLAYGAIN_TRACK_PEAK=", 22))
		{
			for(j = 22, k = 0; j <= strlen(vc->user_comments[i]); j++, k++)
				tag_buf[k] = vc->user_comments[i][j];
			peak_R = atof(tag_buf);
			continue;
		}
		if (!tag_compare(vc->user_comments[i], "RG_RADIO=", 9))
		{
			for(j = 9, k = 0; j <= strlen(vc->user_comments[i]); j++, k++)
				tag_buf[k] = vc->user_comments[i][j];
			gain_R = atof(tag_buf);
			continue;
		}
		if (!tag_compare(vc->user_comments[i], "REPLAYGAIN_TRACK_GAIN=", 22))
		{
			for(j = 22, k = 0; j <= strlen(vc->user_comments[i]); j++, k++)
				tag_buf[k] = vc->user_comments[i][j];
			gain_R = atof(tag_buf);
			continue;
		}
		if (!tag_compare(vc->user_comments[i], "RG_AUDIOPHILE=", 14))
		{
			for(j = 14, k = 0; j <= strlen(vc->user_comments[i]); j++, k++)
				tag_buf[k] = vc->user_comments[i][j];
			gain_A = atof(tag_buf);
			continue;
		}
		if (!tag_compare(vc->user_comments[i], "REPLAYGAIN_ALBUM_GAIN=", 22))
		{
			for(j = 22, k = 0; j <= strlen(vc->user_comments[i]); j++, k++)
				tag_buf[k] = vc->user_comments[i][j];
			gain_A = atof(tag_buf);
			continue;
		}
		if (!tag_compare(vc->user_comments[i], "REPLAYGAIN_ALBUM_PEAK=", 22))
		{
			for(j = 22, k = 0; j <= strlen(vc->user_comments[i]); j++, k++)
				tag_buf[k] = vc->user_comments[i][j];
			peak_A = atof(tag_buf);
			continue;
		}
	}

	/* Do the scale calculations with clipping adjustment, if necessary. */
	if (album)
	{
		if (peak_A)
		{
			peak_A *= 32767.;
			scale_factor = pow(10., gain_A * 0.05);
			factor_clip  = (32767./( peak_A + 1));
			if(scale_factor < factor_clip){
				factor_clip = 1.;
			}else{
				factor_clip /= scale_factor;
			}
			scale_factor *= factor_clip;
		}
		else if (gain_A && peak_R)
		{
			peak_R *= 32767.;
			scale_factor = pow(10., gain_A * 0.05);
			factor_clip  = (32767./( peak_R + 1));
			if(scale_factor < factor_clip){
				factor_clip = 1.;
			}else{
				factor_clip /= scale_factor;
			}
			scale_factor *= factor_clip;
		}
		else
		{
			if(gain_A)
			{
				scale_factor = pow(10., gain_A * 0.05);
			}
		}
	}
	if (title)
	{
		if (peak_R)
		{
			peak_R *= 32767.;
			scale_factor = pow(10., gain_R * 0.05);
			factor_clip  = (32767./( peak_R + 1));
			if(scale_factor < factor_clip){
				factor_clip = 1.;
			}else{
				factor_clip /= scale_factor;
			}
			scale_factor *= factor_clip;
		}
		else
		{
			if(gain_R)
			{
				scale_factor = pow(10., gain_R * 0.05);
			}
		}
	}

exit:
	if (state != NULL)
	{
		vcedit_clear(state);
	}
	if (file != NULL)
	{
		fclose(file);
	}

	return scale_factor;
}

static void usage(void)
{
	fprintf(stderr, " OGGDEC.EXE - Copyright (c) 2002 John Edwards\n\n");
	fprintf(stderr, " Usage: oggdec [options] input.ogg [...]\n\n");
	fprintf(stderr, " OPTIONS (upto THREE options ONLY may be used)\n");
	fprintf(stderr, " -h,  --help       Prints this help information.\n");
	fprintf(stderr, " -a,  --album      Use ReplayGain Album/Audiophile gain settings, OR\n");
	fprintf(stderr, " -r,  --radio      Use ReplayGain Track/Radio gain settings.\n");
	fprintf(stderr, " -d,  --dither X   Dither output, where X =.\n");
	fprintf(stderr, "               0   for       dither OFF (default).\n");
	fprintf(stderr, "               1   for       dither without Noise Shaping.\n");
	fprintf(stderr, "               2   for       dither with Light Noise Shaping.\n");
	fprintf(stderr, "               3   for       dither with Medium Noise Shaping.\n");
	fprintf(stderr, "               4   for       dither with Heavy Noise Shaping.\n");
	fprintf(stderr, " -p,  --play       Plays ogg files thru the soundcard using Windows audio, OR\n");
	fprintf(stderr, " -s,  --shuffle    Shuffles and then plays ogg files using Windows audio.\n");
	fprintf(stderr, " -o,  --stdout     Writes output to stdout in 16 bit ONLY. The format\n");
	fprintf(stderr, "                   options are ignored with this option.\n");
	fprintf(stderr, " PLAYBACK PRIORITY OPTIONS (ONE option ONLY may be used)\n");
	fprintf(stderr, " -c,  --class X    Set playback priority class, where X =\n");
	fprintf(stderr, "              0    for       NORMAL priority.\n");
	fprintf(stderr, "              1    for       HIGH priority (default).\n");
	fprintf(stderr, "              2    for       REALTIME priority.\n");
	fprintf(stderr, " FORMAT OPTIONS (ONE option ONLY may be used)\n");
	fprintf(stderr, " -b,  --bits  X    Set output sample format, where X =\n");
	fprintf(stderr, "              1    for       Unsigned 8 bit PCM data.\n");
	fprintf(stderr, "              2    for       16 bit PCM data (default).\n");
	fprintf(stderr, "              3    for       24 bit PCM data.\n");
	fprintf(stderr, "              4    for       32 bit PCM data.\n");
	fprintf(stderr, "              5    for       32 bit floats.\n\n");
	fprintf(stderr, " INPUT FILES\n");
	fprintf(stderr, " OggDec input files must be Ogg Vorbis I files with\n"); 
	fprintf(stderr, " a sample rate of 48000Hz, 44100Hz, 32000Hz, 24000Hz,\n");
	fprintf(stderr, " 22050Hz, 16000Hz, 12000Hz, 11025Hz or 8000Hz.\n");
	fprintf(stderr, " Wildcards (?, *) can be used in the filename.\n");
}


int main(int argc, char** argv)
{
	OggVorbis_File     vf;
	int                current_section;
	FILE               *ov_input,
	                   *wav_out;
	char               in_path[256],
	                   outfileName[256],
	                   **file_names = NULL;
	int                len,
	                   i,
	                   file_count,
	                   audiophile = 0,
	                   radio = 0,
	                   dithering = 0,
	                   shapingtype = 0,
	                   conv_factor = 65536,             // Default convert to 16 bit PCM
	                   bits,
	                   out_format = 2,
	                   send_to_stdout = 0,
	                   play_files = 0,
	                   shuffle = 0,
	                   short_opt,
	                   option_index = 1;
	double             wrap_prev_pos = 32767,           // Default for 16 bit PCM
	                   wrap_prev_neg = -32768,          // Default for 16 bit PCM
	                   scale;
	unsigned int       pcmbitwidth = 16,                // Default pcmbitwidth
	                   format = WAV_FMT_16BIT,          // Default format 16 bit PCM
	                   play_priority = 1;

	while((short_opt = getopt_long(argc, argv, "hard:psoc:b:", 
					long_options, &option_index)) != -1)
	{
		switch(short_opt)
		{
			case 0:
				fprintf(stderr, "Internal error parsing command line options\n");
				exit(1);
				break;
			case 'h':
				usage();
				exit(0);
				break;
			case 'a':
				audiophile = 1;
				break;
			case 'r':
				radio = 1;
				break;
			case 'd':
				dithering = 1;
   				if(sscanf(optarg, "%d", &shapingtype) != 1)
				{
	    				fprintf(stderr, "Warning: dither type %s not recognised, using default\n", optarg);
		    			break;
				}
				if (shapingtype == 0)
					dithering = 0;
				else if (shapingtype == 1)
					shapingtype = 0;
				else if (shapingtype == 2)
					shapingtype = 1;
				else if (shapingtype == 3)
					shapingtype = 2;
				else if (shapingtype == 4)
					shapingtype = 3;
				break;
			case 'p':
				play_files = 1;
				break;
			case 's':
				play_files = 1;
				shuffle = 1;
				break;
			case 'o':
				send_to_stdout = 1;
				break;
			case 'c':
   				if(sscanf(optarg, "%d", &play_priority) != 1)
				{
	    				fprintf(stderr, "Warning: priority class %s not recognised, using default\n", optarg);
		    			play_priority = 1;
					break;
				}
				if(play_priority > 2)
				{
	    				fprintf(stderr, "Warning: priority class %s not recognised, using default\n", optarg);
		    			play_priority = 1;
				}
				break;
			case 'b':
   				if(sscanf(optarg, "%d", &bits) != 1)
				{
	    				fprintf(stderr, "Warning: output format %s not recognised, using default\n", optarg);
		    			break;
				}
				if (bits == 1)
				{
					pcmbitwidth = 8;
					format = WAV_FMT_8BIT;
					conv_factor = 16777216;
					wrap_prev_pos = 127;
					wrap_prev_neg = -128;
					out_format = 1;
					break;
				}
				else if (bits == 2)
				{
					pcmbitwidth = 16;
					format = WAV_FMT_16BIT;
					conv_factor = 65536;
					wrap_prev_pos = 32767;
					wrap_prev_neg = -32768;
					out_format = 2;
					break;
				}
				else if (bits == 3)
				{
					pcmbitwidth = 24;
					format = WAV_FMT_24BIT;
					conv_factor = 256;
					wrap_prev_pos = 8388607;
					wrap_prev_neg = -8388608;
					out_format = 3;
					break;
				}
				else if (bits == 4)
				{
					pcmbitwidth = 32;
					format = WAV_FMT_32BIT;
					conv_factor = 1;
					wrap_prev_pos = 2147483647;
					wrap_prev_neg = -2147483648;
					out_format = 4;
					break;
				}
				else if (bits == 5)
				{
					pcmbitwidth = 32;
					format = WAV_FMT_FLOAT;
					wrap_prev_pos = 1 - (1 / 2147483648);
					wrap_prev_neg = -1;
					out_format = 5;
					break;
				}
				else
				{
	    				fprintf(stderr, "Warning: output format %s not recognised, using default\n", optarg);
					break;
				}
				break;
		}
	}

	if (play_files)
	{
		pcmbitwidth = 16;
		format = WAV_FMT_16BIT;
		conv_factor = 65536;
		wrap_prev_pos = 32767;
		wrap_prev_neg = -32768;
		out_format = 2;
	}

	Init_Dither (pcmbitwidth, shapingtype);

	if(optind >= argc)
	{
		fprintf(stderr, "ERROR: No input files specified, exiting.\n");
		usage();
		exit(0);
	}
	else
	{
		file_names = argv + optind;
		file_count = argc - optind;
	}

	file_names = glob(file_names, &file_count, shuffle);

	/* Once through the loop for each file */
	for (i = 0; i < file_count; i++)
	{
		vorbis_info   *vi;

		audio_file    *wav_out;

		void          *sample_buffer;
		char          *fileName,
		              percents[64];
		int           eof = 0,
		              infile = 1,
		              outfile = 1,
		              length_secs,
		              length_mins;
		double        bytes_done = 0,
		              file_size;
		__int64       length;

		if (file_names[i][0] == '\0')
		{
			continue;
		}

		/* Let's print some basic information about the decoder and if we are using
		 * ReplayGain tags, which tags are being used. 
		 */
		if(!send_to_stdout)
			fprintf(stderr,VERSION_STRING);

		if(!send_to_stdout && audiophile || radio)
		{
			fprintf(stderr, audiophile 
				? " Running in Album/Audiophile mode\n\n"
				: " Running in Track/Radio mode\n\n");
			/* We are using ReplayGain tags, so get the scale for gain adjustment. */
			scale = get_scale(file_names[i], audiophile, radio);
		}
		else
			scale = 1.0;

		if((ov_input = fopen(file_names[i], "rb")) == NULL)
		{
			fprintf(stderr, "\n ERROR: cannot open %s\n", file_names[i]);
			continue;
		}
		if(ov_open(ov_input, &vf, NULL, 0) < 0)
		{
			fprintf(stderr,"\n Input %s does not appear to be an Ogg bitstream.\n",
						 file_names[i]);
			fclose(ov_input);
			continue;
		}

		vi=ov_info(&vf,-1);
		if(ov_seekable(&vf)) {
			length = ov_pcm_total(&vf, 0);
			file_size = (double)length;
			length *= vi->channels * (pcmbitwidth / 8);
		}

		if(play_files)
		{
			if ( Set_WIN_Params (INVALID_FILEDESC, vi->rate, SAMPLE_SIZE, vi->channels, play_priority) < 0 )
			{
				fprintf (stderr, "\n"VERSION_STRING": Can't access %s\n", "WAVE OUT");
				ov_clear(&vf);
				continue;
			}
		}
		else{
			if(send_to_stdout)
				fileName = NULL;

			else{
				/* Derive the output filename from the input filename. */
				strcpy(outfileName, file_names[i]);
				len = strlen(outfileName);
				while(outfileName[len] != '.') len--;

				if(outfileName[len] == '.')
					outfileName[len] = '\0';

				strcat(outfileName, ".wav");
				fileName = outfileName;
			}
			wav_out = open_output_audio_file(fileName, vi->rate, vi->channels,
				format, OUTPUT_WAV, length);

			if (wav_out == NULL){
				fprintf (stderr, "Not able to open output file: %s\n", fileName) ;
				ov_clear(&vf);
				continue;
			}
		}

		/* Throw the comments plus a few lines about the bitstream we're decoding */
		if(!send_to_stdout)
		{
			char **ptr=ov_comment(&vf,-1)->user_comments;

			while(*ptr)
			{
				fprintf(stderr," %s\n",*ptr);
				++ptr;
			}

			length_secs = ov_time_total(&vf,-1);
			length_mins = length_secs / 60;
			length_secs -= length_mins * 60;
			fprintf(stderr," Encoder Version: %d\n", vi->version);
			fprintf(stderr," Serial Number: %ld\n", ov_serialnumber(&vf, -1));
			fprintf(stderr,"\n Bitstream is %d channel, %ldHz\n", vi->channels, vi->rate);
			fprintf(stderr, " Scale = %6.4lf\n", scale);
			fprintf(stderr," Decoded length: %ld samples = %d:%02d mins.\n", 
					(long)ov_pcm_total(&vf,-1), length_mins, length_secs);
			fprintf(stderr," Encoded by: %s\n",ov_comment(&vf,-1)->vendor);
			fprintf(stderr," Decoding: %s\n\n",file_names[i]);
		}

		while(!eof)
		{
			float **pcm;
       
			long ret = ov_read_float(&vf, &pcm, READ_SIZE, &current_section);

			/* Print out decoding progress. */
			bytes_done += ret;
			if(play_files)
			{
				length_secs = bytes_done / vi->rate;	// time played in secs.
				length_mins = length_secs / 60;
				length_secs -= length_mins * 60;
				fprintf(stderr," Played %d:%02d mins.\r", length_mins, length_secs);
			}
			else if(!send_to_stdout)
			{
				sprintf(percents, "%4.0lf%% decoded.", (bytes_done/file_size)*100);
				fprintf(stderr, " %s\r", percents);
			}

			if (ret == 0) 
			{
				/* EOF */
				eof = 1;
			}
			else if (ret < 0)
			{
				/* error in the stream.  Not a problem, just reporting it in
				 * case we (the app) cares.  In this case, we don't. 
				 */
			}
			else
			{
				/* we don't bother dealing with sample rate changes, etc, but
				 * you'll have to
				 */
				long    j;
				int     convsize=READ_SIZE,
				        k,
				        i = 0,
				        bout=(ret<convsize?ret:convsize);

				/* scale floats to 8, 16, 24 or 32 bit signed ints 
				 * (host order) (unless float output)
				 * and apply ReplayGain scaling, etc. 
				 */
				sample_buffer = malloc(sizeof(float)*vi->channels*bout);
				for(k = 0; k < vi->channels; k++){
					for(j = 0; j < bout; j++, i++){
						__int64 val;
						double Sum;
						pcm[k][j] *= scale;
						if(format != WAV_FMT_FLOAT){
							Sum = pcm[k][j]*2147483647.f;
							if(i > 31)
								i = 0;
							val = dither_output(dithering, shapingtype, i, Sum, k) / conv_factor;
							if (val > (__int64)wrap_prev_pos)
								val = (__int64)wrap_prev_pos;
							else if (val < (__int64)wrap_prev_neg)
								val = (__int64)wrap_prev_neg;
							pcm[k][j] = (float)val;
						}
						else{
							if (pcm[k][j] > wrap_prev_pos)
								pcm[k][j] = wrap_prev_pos;
							else if (pcm[k][j] < wrap_prev_neg)
								pcm[k][j] = wrap_prev_neg;
						}
					}
				}
				sample_buffer = output_to_PCM(pcm, sample_buffer, vi->channels,
						bout, format);
				if(play_files)         // playback
					WIN_Play_Samples (sample_buffer, sizeof(short) * vi->channels * bout);
				else
					write_audio_file(wav_out, sample_buffer, bout * vi->channels);
				free(sample_buffer);
			}
		}
		/* Finished with this file, so cleanup. */
		ov_clear(&vf);
		if(play_files)
			WIN_Audio_close();
		else
			close_audio_file(wav_out, fileName);
	}
	if(!send_to_stdout)
		fprintf(stderr,"\n\n ********** Done decoding all input files. **********\n");
	return(0);
}

/* **************************************** end of oggdec.c ******************************************* */
